#!/usr/bin/env python2.7
# -*- coding: utf-8 -*-

import json
import logging
from os import getenv
from yt.wrapper import YtClient

from common.juggler import ok, warn, crit, send_juggler

'''
SETUP example
simple:
{
  "type": "free",
  "name": "disk_space",
  "value": "24%",
  "level": "crit"
}

complex:
{
  "cluster": "arnold",
  "fields": [
    {
      "type": "free",
      "name": "disk_space",
      "value": "24%",
      "level": "crit"
    },
    {
      "type": "used",
      "name": "disk_space",
      "value": "80%",
      "level": "warn"
    },
    {
      "type": "used",
      "name": "node_count",
      "value": "700",
      "level": "crit"
    },
    {
      "type": "free",
      "name": "node_count",
      "value": "2000",
      "level": "warn"
    },
    {
      "type": "used",
      "name": "chunk_count",
      "value": "2000",
      "level": "warn"
    }
  ]
}

'''

logging.basicConfig(level=logging.INFO)

ATTRIBUTES = ['resource_usage', 'resource_limits']

FIELDS = ['disk_space', 'node_count', 'chunk_count']

LEVELS = {'ok': 0, 'warn': 1, 'crit': 2}
ACTIONS = [ok, warn, crit]
NAMES_BY_LEVEL = {value: key for key, value in LEVELS.items()}


def check(setup, usage, limits):
    levels = {}

    for field in setup.get('fields', [setup]):
        level = LEVELS[field['level']]
        name = field['name']
        used = usage[name]
        limit = limits[name]
        free = field['type'] == 'free'
        edge = field['value']

        if isinstance(edge, str) or isinstance(edge, unicode):
            percentage = '%' in edge
            edge = float(edge.replace('%', ''))
        else:
            percentage = False

        value = limit - used if free else used
        value = float(value) * 100 / limit if percentage else value

        alert = (value < edge) == free

        if alert:
            current_level = levels.get(name, LEVELS['ok'])
            if level > current_level:
                levels[name] = level

    if levels:
        level = max(levels.values())
        message = ', '.join(['%s: %s' % (key, NAMES_BY_LEVEL[value_])
                             for key, value_ in levels.items()])
        return ACTIONS[level](message)
    else:
        return ok()


if __name__ == '__main__':
    token = getenv('YT_TOKEN')
    account = getenv('ACCOUNT')
    setup = json.loads(getenv('SETUP'))
    service_name = getenv('SERVICE')
    juggler_host = getenv('JUGGLER_HOST')

    logging.info('getting resources for %s' % account)

    path = '//sys/accounts/%s/@' % account

    cluster = setup.get("cluster", "hahn")
    client = YtClient(proxy=cluster, token=token)
    data = client.get(path)

    usage = data['resource_usage']
    limits = data['resource_limits']

    logging.info('usage: %s' % usage)
    logging.info('limits: %s' % limits)

    status, text = check(setup, usage, limits)

    send_juggler(juggler_host, service_name, status, text)
