import json

import click

from qloud_deploy.deploy import Deploy
from qloud_deploy.deploy.stage import get_comment, stage_update
from qloud_deploy.docker import DockerRegistry
from qloud_deploy.utils import pprint, print_diff


def click_qloud_command(help):
    def real_decorator(f):
        decorators = [
            click.command(help=help),
            click.argument('stage_ids', nargs=-1, required=True),
            click.option('--token', help='Deploy OAuth token', envvar='DCTL_YP_TOKEN', required=True),
            click.option('--docker-token', help='Docker OAuth token', envvar='QLOUD_TOKEN', required=True),
            click.option('--comment', help='Deployment comment'),
            click.option('--from-json', help='Get environment from json (stdin or filename)', type=click.File('rb')),
            click.option('--image', multiple=True,
                         help='Docker image tag (or `latest` for auto tag). Use <deploy_unit>=<tag> if multiple'),
            click.option('--sandbox',
                         multiple=True,
                         help='Update Sandbox resource by name (<name>:{id|attr}) (pass `remove` for remove)'),
        ]
        func = f
        for decorator in decorators[::-1]:
            func = decorator(func)
        return func
    return real_decorator


@click.group('deploy', help=u"Deploy deployment")
def deploy():
    pass


def get_update_stage(stage_id, token, docker_token, **kwargs):
    deploy_api = Deploy(token)
    current_stage = deploy_api.get_stage(stage_id)
    if kwargs.get('from_json'):
        stage = json.load(kwargs.get('from_json'))
    else:
        registry = DockerRegistry(docker_token)
        stage = stage_update(registry, current_stage, **kwargs)
    return stage, current_stage


@click_qloud_command(help=u"Show active stage (with updates)")
def show(stage_ids, **kwargs):
    for stage_id in stage_ids:
        pprint(get_update_stage(stage_id, **kwargs)[0])


@click_qloud_command(help=u"Deploy stage")
@click.option('-f', '--no-prompt', is_flag=True, help='No prompt before deploy')
@click.option('-u', '--url', is_flag=True, help='Print url after deploy')
@click.option('--downtime-seconds', default=1800, help='Set downtime seconds for deploy (require juggler-token)')
@click.option('--juggler-token', help='Juggler OAuth token', envvar='JUGGLER_OAUTH_TOKEN')
def update(stage_ids, no_prompt, url, downtime_seconds, juggler_token, **kwargs):
    deploy_api = Deploy(kwargs['token'])
    for stage_id in stage_ids:
        stage, current_stage = get_update_stage(stage_id, **kwargs)
        print_diff(current_stage, stage)
        if not no_prompt:
            click.secho("Deploy this stage? [Y/n]: ", fg='red', nl=False)
            answer = click.getchar()
            click.echo()
            if answer != "Y":
                click.secho("Cancel", fg='yellow')
                continue
        pprint(deploy_api.put(stage))
        if url:
            click.echo(deploy_api.get_url(stage_id))
        deploy_api.set_downtime(stage_id, downtime_seconds, get_comment(stage), juggler_token=juggler_token)


deploy.add_command(show)
deploy.add_command(update)
