# -*- coding: utf-8 -*-
import hashlib

from qloud_deploy.api import API


class DockerRegistry(API):
    """Базовый класс для работы с Docker registry"""

    def __init__(self, auth_token):
        self.registry_url = "registry.yandex.net"
        self._cache_tags = {}
        super(DockerRegistry, self).__init__(
            "https://%s/v2/" % self.registry_url,
            {'Authorization': 'OAuth %s' % auth_token},
            json=False
        )

    def _normalize(self, image_url):
        if image_url.startswith(self.registry_url):
            image_url = image_url[len(self.registry_url) + 1:]
        return image_url

    @staticmethod
    def parse_tag(tag):
        try:
            return int(tag.split('_')[0])
        except ValueError:
            return -1

    def get_tags(self, image_name, tags_limit=50):
        image_name = self._normalize(image_name)
        if image_name not in self._cache_tags:
            self._cache_tags[image_name] = sorted(
                self._request('%s/tags/list' % image_name)['tags'],
                key=self.parse_tag
            )[-tags_limit:]
        return self._cache_tags[image_name]

    def get_latest_tag(self, image_name):
        image_name = self._normalize(image_name)
        return self.get_tags(image_name)[-1]

    def get_latest(self, image_name):
        image_name = self._normalize(image_name)
        if not image_name:
            return None, None
        latest_tag = self.get_latest_tag(image_name)
        image_url = "%s/%s:%s" % (self.registry_url, image_name, latest_tag)
        return image_url, self.get_hash(image_url)

    def get_hash(self, image_url):
        image_name = self._normalize(image_url)
        if not image_name:
            return
        image_name, image_tag = image_name.rsplit(':', 1)
        manifest = self._request(
            '%s/manifests/%s' % (image_name, image_tag),
            headers={'Accept': 'application/vnd.docker.distribution.manifest.v2+json'},
            raw=True,
        )
        return 'sha256:%s' % hashlib.sha256(manifest).hexdigest()
