import click

from qloud_deploy.docker import DockerRegistry
from qloud_deploy.utils import pprint, print_warn


@click.group('docker', help=u"Docker info")
def docker():
    pass


@click.command(help=u"List tags")
@click.option('-q', '--qloud-environment-id', help='Get docker images from this qloud environment')
@click.option('-i', '--image', help='Image names')
@click.option('-c', '--component', help='Show just this component')
@click.option('-l', '--limit', default=50, help='Limit listing')
@click.option('--token', help='Qloud OAuth token', envvar='QLOUD_TOKEN', required=True)
def tags(qloud_environment_id, image, component, limit, token):
    registry = DockerRegistry(token)
    image_names = dict()
    if image:
        image_names[''] = image
    if qloud_environment_id:
        from qloud_deploy.qloud import Qloud
        qloud = Qloud(token)
        image_names.update(qloud.get_images(qloud_environment_id))
    if component:
        pprint(registry.get_tags(image_names.get(component, ''), limit))
    else:
        for component_name, image_name in image_names.items():
            print_warn(component_name)
            pprint(registry.get_tags(image_name, limit))


@click.command(help=u"Get latest tag")
@click.option('-q', '--qloud-environment-id', help='Get docker images from this qloud environment')
@click.option('-i', '--image', help='Image name')
@click.option('-c', '--component', help='Show just this component')
@click.option('--show-hash', is_flag=True, help='Show sha256 hash too')
@click.option('--token', help='Qloud OAuth token', envvar='QLOUD_TOKEN', required=True)
def latest(qloud_environment_id, image, component, show_hash, token):
    registry = DockerRegistry(token)
    image_names = dict()
    if image:
        image_names[''] = image
    if qloud_environment_id:
        from qloud_deploy.qloud import Qloud
        qloud = Qloud(token)
        image_names.update(qloud.get_images(qloud_environment_id))
    if component:
        image_name, image_hash = registry.get_latest(image_names.get(component, ''))
        pprint(image_name)
        if show_hash:
            pprint(image_hash)
    else:
        for component_name, image_name in image_names.items():
            print_warn(component_name)
            image_docker_name, image_hash = registry.get_latest(image_name)
            pprint(image_docker_name)
            if show_hash:
                pprint(image_hash)


docker.add_command(tags)
docker.add_command(latest)
