# coding=utf-8
import datetime

import click

from qloud_deploy.infra import Infra


@click.group('infra', help=u"Infra manipulation")
def infra():
    pass


@click.command('create_event', help='Create infra event')
@click.option('--token', help='Infra OAuth token', envvar='INFRA_TOKEN', required=True)
@click.option('-s', '--service', help='Service name', default='disk', type=click.Choice(Infra.service_ids.keys()))
@click.option('--environment', help='Environment name', default='stable')
@click.option('--title', help='Event title', default='Some event')
@click.option('--description', help='Event description', default='')
@click.option('--startTime', help='Event startTime (unixtime). Now, if not defined', type=int)
@click.option('--finishTime', help='Event finishTime (unixtime)', type=int)
@click.option('--duration', help='Event duration (seconds). If finishTime not defined', type=int, default=60*60)
@click.option('--type', help='Event type', type=click.Choice(['issue', 'maintenance']))
@click.option('--severity', help='Event severity', type=click.Choice(['major', 'minor']))
@click.option('--dc', help='Event datacenter (accept `all` keyword)', multiple=True)
@click.option('--ticket', help='Event tickets', multiple=True)
@click.option('--sendEmailNotifications', help='Event sendEmailNotifications', default=True, type=bool)
def create_event(token, duration, **kwargs):
    infra_api = Infra(token)
    event = {}
    for key, value in kwargs.items():
        if key == 'dc':
            if 'all' in value:
                event['setAllAvailableDc'] = True
                continue
            for dc in value:
                event[dc] = True
        elif key == 'ticket':
            event['tickets'] = ','.join(value)
        elif key == 'service':
            event['serviceId'] = infra_api.service_ids[value]
        elif key == 'environment':
            event['environmentId'] = infra_api.environment_ids[kwargs['service']][value]
        else:
            event[key] = value

    if 'startTime' not in event:
        event['startTime'] = int(datetime.datetime.now().strftime('%s'))
    if 'finishTime' not in event:
        event['finishTime'] = event['startTime'] + duration

    result = infra_api.create_event(event)
    click.echo(result)
    click.echo("URL: %s/%s" % (infra_api.event_url, result['id']))


@click.command(help='Create infra event for release')
@click.option('--token', help='Infra OAuth token', envvar='INFRA_TOKEN', required=True)
@click.option('--ticket', help='Release ticket', required=True)
@click.option('--application', help='Application', required=True)
@click.option('--description', help='Description')
@click.option('--severity', help='Severity', default='minor', type=click.Choice(['major', 'minor']))
@click.option('--startTime', help='Start time (unixtime). Now if not defined', type=int)
@click.option('--duration', help='Event duration (seconds). If finishTime not defined', type=int, default=60*60)
@click.pass_context
def release(ctx, token, ticket, application, description, severity, starttime, duration):
    kwargs = {
        'token': token,
        'duration': duration,
        'ticket': [ticket],
        'type': 'maintenance',
        'severity': severity,
        'title': u'Релиз %s' % application,
        'description': description or u'Подробности в тикете %s' % ticket,
        'environment': 'stable',
    }
    if starttime:
        kwargs['startTime'] = starttime

    ctx.invoke(create_event, **kwargs)


infra.add_command(create_event)
infra.add_command(release)
