# -*- coding: utf-8 -*-
import re

import click

from qloud_deploy.docker import DockerRegistry
from qloud_deploy.qloud import Qloud
from qloud_deploy.qloud.environment import environment_update
from qloud_deploy.sandbox import Sandbox
from qloud_deploy.utils import panic, print_warn

MPFS_PACKAGE_TMPL = re.compile(r'^(python-mpfs-(?:disk|queue))?=?(\d+\.\d+.+)$')
MPFS_APP_PACKAGE_MAP = {
    'disk-pworker': 'python-mpfs-disk',
    'disk-mworker': 'python-mpfs-queue',
}


def get_related_package(environment_id, package_or_version):
    """Получить правильный пакет MPFS для указанного env"""
    project, app, _ = environment_id.split('.', 2)
    if project != 'disk':
        panic('Project is not disk')
    if app not in MPFS_APP_PACKAGE_MAP:
        panic('Unknown app. Allowed: %s' % MPFS_APP_PACKAGE_MAP.keys())
    app_related_package_name = MPFS_APP_PACKAGE_MAP[app]
    re_res = MPFS_PACKAGE_TMPL.search(package_or_version)
    if not re_res:
        panic('Not expected package name. %s' % MPFS_PACKAGE_TMPL.pattern)
    package_name, version = re_res.groups()
    if package_name and package_name != app_related_package_name:
        panic('Not valid package name for env "%s". Expect: "%s". Got: "%s"' % (
            environment_id, app_related_package_name, package_name))
    return app_related_package_name, version


@click.group('mpfs', help=u"MPFS deploy tools")
def mpfs():
    pass


@mpfs.command('deploy_package', help='Deploy package for env')
@click.argument('environment_id')
@click.argument('package_or_version')
@click.option('--token', required=True, help='Qloud OAuth token', envvar='QLOUD_TOKEN')
@click.option('--wait', help='Wait until env deployed', is_flag=True)
@click.option('--ticket', help='Release ticket')
@click.option('--latest-env-version', is_flag=True, help='Use latest env version, even if it is unstable')
@click.option('--juggler-token', help='Juggler OAuth token', envvar='JUGGLER_OAUTH_TOKEN')
def deploy_package(environment_id, package_or_version, token=None,
                   wait=True, ticket="", latest_env_version=False, juggler_token=None):
    """Получить из sandbox id ресурса, связанного с MPFS пакетом, и передеплоить с ним env"""
    package_name, version = get_related_package(environment_id, package_or_version)
    package = '%s=%s' % (package_name, version)
    sandbox = Sandbox()
    resource = sandbox.get_package_related_resource(package_name, version)
    package_resource_id = resource['id']
    click.echo('Got mpfs related resource "%s" for package "%s"' % (package_resource_id, package))
    qloud = Qloud(token)
    if not qloud.is_deployed(environment_id):
        print_warn('Env %s not in "deployed" status' % environment_id)

    current_environment_version = qloud.get_stable_environment_version(environment_id, latest_env_version)
    current_environment = qloud.dump(environment_id, current_environment_version)
    kwargs = {
        'comment': 'Update MPFS resource (sandbox id: "%s") for package %s %s' % (
            package_resource_id, package, ticket or ''
        ),
        'mpfs': package_resource_id,
    }
    environment = environment_update(
        qloud,
        DockerRegistry(token),
        current_environment,
        environment_id,
        strict=True,
        **kwargs
    )
    qloud.upload(environment)
    click.echo('Env "%s" updated' % environment_id)
    qloud.set_downtime(environment_id, 1800, environment['comment'], juggler_token=juggler_token)
    if wait:
        click.echo('Wait until "%s" deployed' % environment_id)
        qloud.wait_until_deployed(environment_id)
