# -*- coding: utf-8 -*-
import os
import time

from qloud_deploy.api import API
from qloud_deploy.errors import Timeout
from qloud_deploy.juggler import Juggler
from qloud_deploy.sandbox import Sandbox
from qloud_deploy.utils import print_warn


class QloudEnvStatus(object):
    deployed = 'DEPLOYED'


class Qloud(API):
    """
    Базовый класс работы с Qloud API
    """
    def __init__(self, auth_token):
        self.public_url = "https://platform.yandex-team.ru"
        self.unversioned_url = self.public_url + "/api/"
        super(Qloud, self).__init__(
            self.unversioned_url + "v1/",
            {'Authorization': 'OAuth %s' % auth_token},
        )

    def get_url(self, environment_id):
        return "%s/projects/%s" % (self.public_url, environment_id.replace('.', '/'))

    def get_stable_environment_version(self, environment_id, use_latest_version=False):
        environment = self._request('environment/stable/%s' % environment_id)
        if use_latest_version:
            return environment['versions'][0]['version']
        else:
            return environment['version']

    def get_last_environment_versions(self, environment_id, last_versions=None):
        environment = self._request('environment/stable/%s' % environment_id)
        versions = [x['version'] for x in environment['versions']]
        if last_versions:
            return versions[:last_versions]
        else:
            return versions

    def get_detail(self, environment, environment_version=None):
        if environment_version is None:
            return self._request('platform/detail/%s' % environment)
        else:
            return self._request('platform/detail/%s/%s' % (environment, environment_version))

    def dump(self, environment_id, environment_version=None):
        if environment_version is None:
            return self._request('environment/dump/%s' % environment_id)
        else:
            return self._request('environment/dump/%s/%s' % (environment_id, environment_version))

    def get_images(self, environment_id):
        environment = self.dump(environment_id)
        return {
            component['componentName']: component['properties']['repository'].split(':')[0]
            for component in environment['components']
            if component.get('componentType') == 'standard'
        }

    def upload(self, environment):
        return self._request('environment/upload/return-header', method='POST', json_data=environment)

    def get_status(self, environment_id):
        return self._request('environment/status/%s' % environment_id)

    def is_deployed(self, environment_id):
        status_dict = self.get_status(environment_id)
        return status_dict['status'] == QloudEnvStatus.deployed

    def wait_until_deployed(self, environment_id, polling_timeout=600, polling_interval=2):
        """Дождаться, пока env не будет в состоянии deployed"""
        start_ts = time.time()
        while True:
            if self.is_deployed(environment_id):
                return
            if start_ts - time.time() > polling_timeout:
                raise Timeout()
            time.sleep(polling_interval)

    @staticmethod
    def http_stop_action(method='GET', path='/container_stop_hook', port=3199, timeout=300000):
        hook_settings = {
            'httpMethod': method,
            'httpPath': path,
            'httpPort': port,
            'timeoutMs': timeout,
            'type': 'http'
        }
        return hook_settings

    @staticmethod
    def http_status_hook(path='/ping', port=80, timeout=1000):
        hook_settings = {
            'path': path,
            'port': port,
            'timeout': timeout,
            'type': 'http'
        }
        return hook_settings

    def http_service_status_action(self):
        return self.http_status_hook(path='/container_status_hook', port=3199)

    @staticmethod
    def update_status_hooks(new_hook, hooks):
        """
        Добавить status_hook по path и port, если отсутствует
        """
        if new_hook['type'] != 'http':
            return hooks

        for hook in hooks:
            if hook['path'] == new_hook['path'] and hook['port'] == new_hook['port']:
                hook.update(new_hook)  # На случай, если обновился таймаут
                break
        else:
            hooks.append(new_hook)

        return hooks

    @staticmethod
    def remove_status_hook(hooks, port=None, path=None):
        """
        Удалить статус хук по port и path
        """
        return [h for h in hooks if (port is None or h['port'] != port) or (path is None or h['path'] != path)]

    @staticmethod
    def update_secrets(secrets, secret_name, secret_target):
        """
        Обновить секреты.
        Если отсутствует - добавить
        Если есть - обновить target и включить usage
        """
        if not secret_name.startswith('secret.'):
            secret_name = 'secret.' + secret_name

        for secret in secrets:
            if secret['objectId'] == secret_name:
                secret['target'] = secret_target
                secret['used'] = True
                break
        else:
            secrets.append({'objectId': secret_name, 'target': secret_target, 'used': True})

        return secrets

    @staticmethod
    def update_sandbox(name, package_or_id, resources, strict=False):
        """
        Обновить Sandbox ресурс на новый id.
        Если ресурс отсутствует - добавляет его.
        Если package_or_id == "remove" - удаляет ресурс из окружения
        """
        path = '/%s.tar.gz' % name
        if package_or_id == "remove":
            return [r for r in resources if r['symlink'] != path]
        try:
            new_id = Sandbox().get_resource_by_key(package_or_id)['id']
        except ValueError:
            if strict:
                raise
            return resources
        for resource in resources:
            if resource['symlink'] == path:
                resource['id'] = new_id
                break
        else:
            if strict:
                raise ValueError('"%s" resource not found for current env.' % name)
            resources.append({
                'dynamic': False,
                'extract': False,
                'id': new_id,
                'localName': os.path.basename(path),
                'symlink': path,
            })
        return resources

    def get_project_tree(self):
        return self._request('project-tree', base_url=self.unversioned_url)

    def get_applications(self, project="disk", applications=None):
        """Все приложения в проекте"""
        project_tree = list(filter(lambda p: p['projectName'] == project, self.get_project_tree()['projects']))[0]
        project_applications = project_tree['applications']
        if applications:
            project_applications = list(filter(lambda app: app['name'] in applications, project_applications))
        return project_applications

    def get_all_environment_ids(self, project="disk", applications=None, environments=()):
        """Список id всех окружений в проекте"""
        def environment_startswith(s):
            for environment in environments:
                if s.split('.')[-1].startswith(environment):
                    return True
            return False

        all_env_ids = [e['objectId'] for a in self.get_applications(project, applications) for e in a['environments']]
        if environments:
            all_env_ids = list(filter(environment_startswith, all_env_ids))
        return all_env_ids

    @staticmethod
    def set_downtime(environment_id, seconds, comment, juggler_token=None, dt_services=()):
        if not juggler_token:
            print_warn("Juggler OAuth token not set. SKIP downtime")
            return

        dt_services = dt_services or ['service-locker', 'service-status']

        filters = [
            {
                'tags': [
                    'qloud-ext.' + environment_id,
                ],
                'service': service,
            }
            for service in dt_services
        ]
        Juggler(juggler_token).set_downtimes(
            filters,
            seconds=seconds,
            description=comment,
        )
