import json

import click

from qloud_deploy.docker import DockerRegistry
from qloud_deploy.qloud import Qloud
from qloud_deploy.qloud.environment import environment_update
from qloud_deploy.report import Reporter
from qloud_deploy.utils import pprint, print_diff


def click_qloud_command(help):
    def real_decorator(f):
        decorators = [
            click.command(help=help),
            click.argument('environment_ids', nargs=-1, required=True),
            click.option('--token', help='Qloud OAuth token', envvar='QLOUD_TOKEN', required=True),
            click.option('--comment', help='Deployment comment'),
            click.option('--disk-size', help='Disk size (GB)'),
            click.option('--from-json', help='Get environment from json (stdin or filename)', type=click.File('rb')),
            click.option('--image', multiple=True,
                         help='Docker image (full registry path or `latest` for auto tag). '
                              'Use <component>=<image> if multiple'),
            click.option('--size', help='Instance size'),
            click.option('--network', help='Instance network'),
            click.option('--stop-hook', is_flag=True, help='Enable default stop hook'),
            click.option('--status-hook', help='Add status hook (PORT:PATH:TIMEOUT)'),
            click.option('--service-status-hook', is_flag=True, help='Enable default service-status hook'),
            click.option('--remove-status-hook-by-port', multiple=True, help='Remove status hook by port'),
            click.option('--no-unistat', is_flag=True, help='Enable unistat'),
            click.option('--unistat-port', help='Enable unistat on port'),
            click.option('--unistat-path', help='Enable unistat with path'),
            click.option('--units', help='Location units (format: `LOCATION:NUM`)', multiple=True),
            click.option('--user-var', help='Set user environment variable', multiple=True),
            click.option('--application', help='Application Sandbox ID (pass `remove` for remove)'),
            click.option('--mpfs', help='MPFS Sandbox ID (pass `remove` for remove)'),
            click.option('--settings', help='Settings Sandbox ID (pass `remove` for remove)'),
            click.option('--secret', help='Add secrets by id:path', multiple=True),
            click.option('--latest-env-version', is_flag=True, help='Use latest env version, even if it is unstable'),
            click.option('--molly', is_flag=True, help='Enable molly mirror in route settings'),
        ]
        func = f
        for decorator in decorators[::-1]:
            func = decorator(func)
        return func
    return real_decorator


@click.group('qloud', help=u"Qloud deployment")
def qloud():
    pass


def get_update_environment(environment_id, token, **kwargs):
    qloud_api = Qloud(token)
    registry = DockerRegistry(token)
    environment_version = qloud_api.get_stable_environment_version(environment_id, kwargs['latest_env_version'])
    current_environment = qloud_api.dump(environment_id, environment_version)

    if kwargs.get('from_json'):
        environment = json.load(kwargs.get('from_json'))
    else:
        environment = environment_update(qloud_api, registry, current_environment, environment_id, **kwargs)
    return environment, current_environment


@click_qloud_command(help=u"Show active environment (with updates)")
def show(environment_ids, **kwargs):
    for environment_id in environment_ids:
        pprint(get_update_environment(environment_id, **kwargs)[0])


@click_qloud_command(help=u"Deploy environment")
@click.option('-f', '--no-prompt', is_flag=True, help='No prompt before deploy')
@click.option('-u', '--url', is_flag=True, help='Print url after deploy')
@click.option('--downtime-seconds', default=1800, help='Set downtime seconds for deploy (require juggler-token)')
@click.option('--juggler-token', help='Juggler OAuth token', envvar='JUGGLER_OAUTH_TOKEN')
def deploy(environment_ids, no_prompt, url, downtime_seconds, juggler_token, **kwargs):
    qloud_api = Qloud(kwargs['token'])
    for environment_id in environment_ids:
        environment, current_environment = get_update_environment(environment_id, **kwargs)
        print_diff(current_environment, environment)
        if not no_prompt:
            click.secho("Deploy this environment? [Y/n]: ", fg='red', nl=False)
            answer = click.getchar()
            click.echo()
            if answer != "Y":
                click.secho("Cancel", fg='yellow')
                continue
        pprint(qloud_api.upload(environment))
        if url:
            click.echo(qloud_api.get_url(environment_id))
        qloud_api.set_downtime(environment_id, downtime_seconds, environment['comment'], juggler_token=juggler_token)


@click_qloud_command(help=u"Environment status")
def status(environment_ids, **kwargs):
    for environment_id in environment_ids:
        pprint(Qloud(kwargs['token']).get_status(environment_id))


@click.command('get_all_environments', help="Get all environments")
@click.option('--token', help='Qloud OAuth token', envvar='QLOUD_TOKEN', required=True)
@click.option('-a', '--application', help='Filter by application', multiple=True)
@click.option('-p', '--project', help='Filter by project (default: disk)', multiple=True, default=['disk'])
@click.option('-e', '--environment', help='Filter by environment (substring)', multiple=True)
@click.option('-s', '--to-string', help='Join to string', is_flag=True)
def get_all_environments(token, application, project, environment, to_string=False):
    qloud_api = Qloud(token)
    result = []
    for p in project:
        result.extend(qloud_api.get_all_environment_ids(p, set(application), environment))
    if to_string:
        click.echo(" ".join(result))
    else:
        pprint(result)


@click.command('get_all_applications', help="Get all applications")
@click.option('--token', help='Qloud OAuth token', envvar='QLOUD_TOKEN', required=True)
@click.option('-p', '--project', help='Filter by project (default: disk)', multiple=True, default=['disk'])
@click.option('-s', '--to-string', help='Join to string', is_flag=True)
@click.option('-n', '--names-only', help='Print only names', is_flag=True)
def get_all_applications(token, project, to_string=False, names_only=False):
    qloud_api = Qloud(token)
    result = []
    for p in project:
        applications = qloud_api.get_applications(p)
        if names_only:
            applications = [a['name'] for a in applications]
        result.extend(applications)
    if to_string:
        click.echo(" ".join(result))
    else:
        pprint(result)


@click.command(help=u"Generate xlsx report")
@click.option('--token', help='Qloud OAuth token', envvar='QLOUD_TOKEN', required=True)
@click.option('-n', '--nversions', help='Report last N versions, 0 = no limit', default=10)
@click.option('-e', '--environment', help='Reportable environments', multiple=True, required=True)
@click.option('-f', '--filename', help='Report filename')
@click.option('-l', '--logscale', help='Use logarithm scale for charts', is_flag=True)
def xlsx(token, versions, environment, filename=None, logscale=False):
    environments = ','.join(environment)
    rep = Reporter(token, ("ACTIVATED", "PREPARED", "CONFIGURED", "ALLOCATED"))
    report = rep.generate_xls(environments, versions, log_scale=logscale, filename=filename)
    click.echo('Report generated: %s' % report)


qloud.add_command(show)
qloud.add_command(deploy)
qloud.add_command(status)
qloud.add_command(get_all_environments)
qloud.add_command(get_all_applications)
qloud.add_command(xlsx)
