# -*- coding: utf-8 -*-

from datetime import datetime

import xlsxwriter
from qloud_deploy.utils import print_warn


def version_to_time(version, sformat='%Y-%m-%d %H:%M:%S'):
    dt = datetime.fromtimestamp(float(version) / 1000)
    return dt.strftime(sformat)


class Reporter:

    def __init__(self, token, states, ):
        from qloud_deploy.qloud import Qloud
        self.qloud = Qloud(token)
        self.states = states

    def extract_details(self, environment, version=None):
        # Exctract detailed data from qloud per environment
        data = {}
        detail = self.qloud.get_detail(environment, version)
        processes = detail['processes']
        for proc in processes:
            state = proc['currentState']
            if state in self.states:
                start, end = map(
                    lambda date: datetime.strptime(date.split('+')[0], '%Y-%m-%dT%H:%M:%S.%f'),
                    (proc['started'], proc['completed']),
                )
                data[state] = (end - start).total_seconds()
        return data

    def get_last_versions_data(self, component, n=None):
        # Get last N component versions
        data = []
        try:
            versions = reversed(self.qloud.get_last_environment_versions(component, n))  # time ascending order
        except Exception as E:
            print_warn(E.__repr__())
            return data

        for ver in versions:
            try:
                data.append({'version': ver, 'date': version_to_time(ver),
                             'timings': self.extract_details(component, ver)})
            except Exception as E:
                print_warn(E.__repr__())

        return data

    def get_time_by_date_per_state(self, data):
        # Angry analytics process
        res = {}
        for prj, versions in data.items():
            prj_data = [['version', 'date'] + list(self.states)]
            for ver in versions:
                state_list = []
                for state in self.states:
                    try:
                        state_list.append(ver['timings'][state])
                    except KeyError as E:
                        print_warn('; '.join(map(str, (E.__repr__(), prj, ver, state))))
                        state_list = []
                        break
                if state_list:
                    prj_data.append([ver['version'], ver['date']]+state_list)
            res[prj] = prj_data
        return res

    def generate_xls(self, components, versions, log_scale=False, filename=None):
        # XLSX creator, returns generated filename
        comps = set(components.strip().split(','))
        data = {component: self.get_last_versions_data(component, versions) for component in comps}

        if not filename:
            filename = 'qloud_report_{}.xlsx'.format(datetime.now().strftime('%Y-%m-%d_%H:%M:%S'))

        workbook = xlsxwriter.Workbook(filename)
        sheets = {k: workbook.add_worksheet(k) for k in ('charts', 'data')}

        x = 0
        chart_x = 0
        for component, dataframe in sorted(self.get_time_by_date_per_state(data).items(), key=lambda x: x[0]):
            if not dataframe:
                continue

            # Write component name
            sheets['data'].write_row(x, 0, [component])
            x += 1
            first_row = x+1
            y = 0
            for row in dataframe:
                sheets['data'].write_row(x, 0, row)
                x += 1
                y = max(y, len(row))

            chart = workbook.add_chart({'type': 'line'})
            chart.set_title({'name': component})
            chart.set_x_axis({'date_axis': True,
                              'major_gridlines': {'visible': True, 'line': {'width': 0.75, 'dash_type': 'dash'}}})
            y_axis = {'name': 'sec'}
            if log_scale:
                y_axis.update({'log_base': 2})
            chart.set_y_axis(y_axis)

            # must skip 1 col for get_time_by_date_per_state
            SKIP = 1
            for c, state in zip(range(1, y), self.states):
                chart.add_series({
                    'categories': ['data', first_row, SKIP, x-1, SKIP],
                    'values': ['data', first_row, SKIP+c, x-1, SKIP+c],
                    'marker': {'type': 'automatic'},
                    'name': state
                })
            sheets['charts'].insert_chart(chart_x, 0, chart, {'x_scale': 2, 'y_scale': 1.33})
            chart_x += 20
            # Skip 2 rows for visibility
            x += 2

        workbook.close()
        return filename
