# -*- coding: utf-8 -*-
import json

from qloud_deploy.api import API
from qloud_deploy.errors import SandboxError


class Sandbox(API):
    """Базовый класс для работы с Sandbox API"""

    def __init__(self):
        super(Sandbox, self).__init__("https://sandbox.yandex-team.ru/api/v1.0/")
        self._cached_resources = {}

    def get_resource_by_key(self, key):
        if isinstance(key, (str, unicode)) and '=' in key:
            package_name, package_version = key.split('=', 1)
            return self.get_package_related_resource(package_name, package_version)
        else:
            sandbox_id = int(key)
            return self.get_resources(id=[sandbox_id], resource_type='')[0]

    def get_package_related_resource(self, package_name, package_version):
        """Получить соответсвующий пакету sandbox resource"""
        resources = self.get_package_resources(package_name, package_version=package_version)
        if not resources:
            raise SandboxError('Package related resource not found. "%s %s"' % (package_name, package_version))
        elif len(resources) > 1:
            raise SandboxError('Find %i package related resources. %s' % (len(resources), resources))
        return resources[0]

    def get_resources(self, limit=10, resource_type='DISK_COMPRESSED_RESOURCE_APPLICATION',
                      owner='DISK-ADMIN', state='READY', **kwargs):
        params = {
            'limit': limit,
            'state': state,
            'type': resource_type,
            'owner': owner,
        }
        if kwargs:
            params.update(kwargs)
        params_key = str(params)
        if params_key not in self._cached_resources:
            self._cached_resources[params_key] = self._request('resource', params=params)['items']
        return self._cached_resources[params_key]

    def get_package_resources_by_attribute(self, package_name, package_version="", limit=10, **kwargs):
        """Рекомендуемый способ, быстрый поиск"""
        return self.get_resources(limit=limit, attrs=json.dumps({package_name: package_version}), **kwargs)

    def get_package_resources_by_description(self, package_name, package_version="", limit=10, **kwargs):
        """
        BACKWARD COMPABILITY!
        Выгружает список всех ресурсов за пятикратный лимит и ищет пакет по description.
        Рекомендуется переделать на схему с аттрибутами.
        """
        all_resources = self.get_resources(limit=limit * 5, **kwargs)
        filtered_resources = []
        for resource in all_resources:
            if package_name in resource['description'] and package_version in resource['description']:
                filtered_resources.append(resource)

        return filtered_resources[:limit]

    def get_package_resources(self, package_name, package_version="", limit=10, **kwargs):
        resources = self.get_package_resources_by_attribute(package_name, package_version, limit, **kwargs)
        if resources:
            return resources
        return self.get_package_resources_by_description(package_name, package_version, limit, **kwargs)

    @staticmethod
    def find_package_version(resource):
        """Ищет пакет в аттрибутах, либо в описании. Необходимо из-за get_package_resources_by_description"""
        attr_package_list = [k for k in resource['attributes'].keys() if k not in ('ttl', 'backup_task')]
        if attr_package_list:
            package = attr_package_list[0]
            version = resource['attributes'][package]
        else:
            try:
                package, version = resource['description'].strip('" ').replace('=', ' ').split()[:2]
            except ValueError:
                package, version = resource['description'], ''
        return package, version

    def format_resources(self, resources):
        """Выкинуть все неинтересные поля, оставив только человекочитаемые"""
        formatted_resources = []
        for resource in resources:
            package, version = self.find_package_version(resource)
            formatted_resource = {
                'id': resource['id'],
                'created': resource['time']['created'],
                'package': package,
                'version': version,
            }
            formatted_resources.append(formatted_resource)
        return formatted_resources
