#!/usr/bin/env python
# -*- encoding: utf-8 -*-

import logging
import os

from slack import SlackClient
from telegram import TelegramClient
from telegraph import TelegraphClient
from yandex_abc import YandexABCClient
from yandex_calendar import YandexCalendarClient
from yandex_staff import YandexStaffClient


logging.basicConfig(
    level=logging.INFO,
    format="tskv\ttskv_format=ydisk-sandbox-default-log\tname=%(name)s\tunixtime=%(created)d\t"
           "timestamp=%(asctime)s\ttimezone=+0300\tlevel=%(levelname)s\tmessage=%(message)s")
logger = logging.getLogger('robot-switcher')


def get_current_admin_on_duty(forward_number, admins):
    for admin in admins:
        if forward_number in ("%s" % admin['work_phone'], "55%s" % admin['work_phone']):
            return admin
        for phone in admin['phones']:
            if phone['number'] == forward_number:
                return admin
    return None


def get_next_admin_on_duty(duty, admins):
    for admin in admins:
        if admin['login'] in duty:
            return admin
    return None


def get_new_duty_from_calendar(yandex_staff_client):
    admins = yandex_staff_client.get_group_members(os.environ['ADMIN_DEPARTMENT'])
    yandex_calendar_client = YandexCalendarClient(
        os.environ['CALENDAR_PRIVATE_TOKEN'], os.environ.get('TZ', 'Europe/Moscow'))
    if bool(os.environ.get('CALENDAR_PREVIOUS_DUTY', False)):
        event_duty, event_date = yandex_calendar_client.get_previous_distinct_event()
    else:
        event_duty, event_date = yandex_calendar_client.get_current_event()
    if not event_duty:
        logger.error("Calendar is empty. No switch")
        return None, None
    next_duty_admin = get_next_admin_on_duty(event_duty, admins)
    if next_duty_admin is None:
        logger.error("No next admin")
        return None, None
    logger.info("Next admin on duty: %s", next_duty_admin)
    if not yandex_calendar_client.is_today(event_date):
        logger.info("To early")
        return None, None
    return next_duty_admin['login'], yandex_staff_client.get_call_number(next_duty_admin)


def get_duty_from_abc(yandex_staff_client):
    abc_client = YandexABCClient(os.environ['ABC_TOKEN'])
    schedule__slug = os.environ.get('ABC_SCHEDULE_SLUG', None)
    if bool(os.environ.get('CALENDAR_PREVIOUS_DUTY', False)):
        current_abc_duty = abc_client.get_previous_duty(os.environ['ABC_SERVICE_ID'], schedule__slug=schedule__slug)
    else:
        current_abc_duty = abc_client.get_duty(os.environ['ABC_SERVICE_ID'], schedule__slug=schedule__slug)
    if not current_abc_duty:
        logger.error("ABC duty is empty. No switch")
        return None, None
    logger.info("Next admin on duty: %s", current_abc_duty)
    if not YandexCalendarClient.is_day():
        logger.info("To early")
        return None, None
    return current_abc_duty, yandex_staff_client.get_worker_phone_number(current_abc_duty)


def set_slack_topic(slack_client, login, message_prefix=None):
    try:
        current_login = slack_client.extract_login_from_topic()
    except:
        pass
    else:
        if current_login == login:
            return

    slack_client.set_channel_topic(login, message_prefix)
    for channel_id in filter(len, os.environ.get('SLACK_ADDITIONAL_CHANNELS', '').split(',')):
        slack_client.set_channel_topic(login, message_prefix, channel_id=channel_id)


def main():
    try:
        worker_phone_number = None
        slack_client = SlackClient(os.environ.get('SLACK_TOKEN'), os.environ.get('SLACK_CHANNEL'))
        telegram_client = TelegramClient(os.environ.get('TELEGRAM_BOT_TOKEN'))
        yandex_staff_client = YandexStaffClient(os.environ['STAFF_TOKEN'])
        robot_phone = os.environ.get('ROBOT_PHONE')
        if not robot_phone:
            if os.environ.get('FORCE_ADMIN'):
                login = os.environ['FORCE_ADMIN']
                logger.info('Forced duty admin: %s', login)
            else:
                login, _ = get_new_duty_from_calendar(yandex_staff_client)
            if not login:
                logger.info('New duty login is unknown. Nothing changed')
                return

            old_login = slack_client.extract_login_from_topic()
            logger.info('Duty login extracted from topic: "%s"', old_login)
            if old_login == login:
                logger.info('Login is equal to old_login. Nothing changed')
            else:
                if os.environ.get('DRY_RUN_SLACK', 'no') != 'yes':
                    set_slack_topic(slack_client, login, os.environ.get('SLACK_TOPIC_PREFIX', None))
        else:
            telegraph_client = TelegraphClient(os.environ.get('TELEGRAPH_TOKEN'))
            forward_number = telegraph_client.get_forward(robot_phone)
            if os.environ.get('FORCE_ADMIN'):
                login = os.environ['FORCE_ADMIN']
                worker_phone_number = yandex_staff_client.get_worker_phone_number(login)
                logger.info('Forced duty admin: %s - %s', login, worker_phone_number)
            elif os.environ.get('SLACK_PREFERRED', 'no') == 'yes':
                login = slack_client.extract_login_from_topic()
                logger.info('Duty login extracted from topic: "%s"', login)
                worker_phone_number = yandex_staff_client.get_worker_phone_number(login)
            elif os.environ.get('TELEGRAM_PREFERRED', 'no') == 'yes':
                tg_username = telegram_client.extract_login_from_topic(chat_id=os.environ.get('TELEGRAM_CHAT_ID'))
                login = yandex_staff_client.get_login_by_telegram(tg_username)
                logger.info('Duty login extracted from topic: "%s"', login)
                worker_phone_number = yandex_staff_client.get_worker_phone_number(login)
            elif os.environ.get('ABC_PREFERRED', 'no') == 'yes':
                login, worker_phone_number = get_duty_from_abc(yandex_staff_client)
            else:
                login, worker_phone_number = get_new_duty_from_calendar(yandex_staff_client)

            if worker_phone_number and worker_phone_number != forward_number:
                logger.info("Set new number %s" % worker_phone_number)
                if os.environ.get('DRY_RUN_REDIRECT', 'no') != 'yes':
                    telegraph_client.change_forward(os.environ['ROBOT_PHONE'], worker_phone_number)
            else:
                logger.info("Duty not changed")

            if login:
                if os.environ.get('DRY_RUN_SLACK', 'no') != 'yes' and os.environ.get('SLACK_PREFERRED', 'no') != 'yes':
                    if bool(os.environ.get('CALENDAR_PREVIOUS_DUTY', False)):
                        set_slack_topic(slack_client, login, 'Второй дежурный сегодня ')
                    else:
                        set_slack_topic(slack_client, login, os.environ.get('SLACK_TOPIC_PREFIX', None))
                if os.environ.get('DRY_RUN_TELEGRAM', 'no') != 'yes' and os.environ.get('TELEGRAM_BOT_TOKEN'):
                    tg_username = yandex_staff_client.get_telegram_username(login)
                    if os.environ.get('TELEGRAM_ONE_CHAT', 'no') == 'yes':
                        for telegram_chat_id in os.environ.get('TELEGRAM_CHAT_IDS').split(','):
                            telegram_client.update_duty_in_pinned_message(
                                os.environ.get('TELEGRAM_DUTY_PREFIX'),
                                tg_username,
                                chat_id=telegram_chat_id,
                            )
                    else:
                        telegram_chat_id = os.environ.get('TELEGRAM_CHAT_ID')
                        tg_current_username = telegram_client.extract_login_from_topic(chat_id=telegram_chat_id)
                        if tg_username != tg_current_username:
                            telegram_client.set_channel_topic(
                                tg_username,
                                os.environ.get('TELEGRAM_TOPIC_PREFIX', None),
                                chat_id=telegram_chat_id,
                            )
            else:
                logger.info("No change")
    except Exception as e:
        logger.error(e)
        raise
