# -*- encoding: utf-8 -*-
import logging
import os
import re
import requests

from utils import generate_duty_message, log_response


class SlackClient(object):
    URL = 'https://slack.com/api/'

    def __init__(self, token, channel_id):
        self.token = token
        self.channel_id = channel_id
        self.logger = logging.getLogger('YandexSlackClient')
        self.headers = {
            'user-agent': 'disk.robot_switcher (GSID: %s)' % os.getenv('GSID', '').replace('\n', ' '),
        }

    def get_channel_topic(self, channel_id=None):
        channel_id = channel_id or self.channel_id
        self.logger.info('Get channel topic, id="%s"', channel_id)
        url = SlackClient.URL + 'conversations.info'
        data = {'token': self.token, 'channel': channel_id}
        response = requests.post(url, data=data, headers=self.headers)
        log_response(self.logger, response)
        if response.status_code != 200:
            raise RuntimeError('Unable to get channel topic: %d %s' % (response.status_code, response.text))
        return response.json()['channel']['topic']['value']

    def set_channel_topic(self, login, message_prefix=None, channel_id=None):
        message_prefix = message_prefix or 'Дежурный сегодня '
        channel_id = channel_id or self.channel_id
        self.logger.info('Set channel topic, id="%s"', channel_id)
        url = SlackClient.URL + 'conversations.setTopic'
        data = {
            'token': self.token,
            'channel': channel_id,
            'topic': message_prefix + ":{login}: @{login}".format(login=login)
        }
        response = requests.post(url, data=data, headers=self.headers)
        log_response(self.logger, response)
        if response.status_code != 200:
            raise RuntimeError('Unable to set conversation topic: %d %s' % (response.status_code, response.text))
        return response.json()

    def send_message(self, login, message=None):
        self.logger.info('Send message to %s', login)
        url = SlackClient.URL + 'chat.postMessage'
        if message is None:
            message = generate_duty_message()
        data = {
            'token': self.token,
            'channel': '@%s' % login,
            'text': message,
        }
        response = requests.post(url, data=data, headers=self.headers)
        log_response(self.logger, response)
        if response.status_code != 200:
            raise RuntimeError('Unable to send message: %d %s' % (response.status_code, response.text))
        return response.json()

    def extract_login_from_topic(self):
        topic = self.get_channel_topic()
        result = re.search('<@([^>]+)>', topic)
        if result:
            user_id = result.group(1)
            return self.get_user_name(user_id)
        result = re.search('@\S+', topic)
        if result:
            return result.group()[1:]  # skip `@`
        return None

    def get_user_name(self, user_id):
        self.logger.info('Get user name, id="%s"', user_id)
        url = SlackClient.URL + 'users.info'
        data = {'token': self.token, 'user': user_id}
        response = requests.post(url, data=data, headers=self.headers)
        log_response(self.logger, response)
        if response.status_code != 200:
            raise RuntimeError('Unable to get user name: %d %s' % (response.status_code, response.text))
        return response.json()['user']['name']
