# -*- encoding: utf-8 -*-
import logging
import os
import re
import requests

from utils import log_response


class TelegramClient(object):
    URL = 'https://api.telegram.org/bot'

    def __init__(self, token, chat_id=None):
        self.token = token
        self.chat_id = chat_id
        self.logger = logging.getLogger('YandexTelegramClient')
        self.headers = {
            'user-agent': 'disk.robot_switcher (GSID: %s)' % os.getenv('GSID', '').replace('\n', ' '),
        }

    def url(self, command):
        if not self.token:
            return self.URL
        return self.URL + self.token + '/' + command

    def http_post(self, *args, **kwargs):
        try:
            response = requests.post(*args, **kwargs)
        except Exception as e:
            raise RuntimeError('Error telegram requests.post: %s', str(e).split(':')[0])
        else:
            log_response(self.logger, response)
            if response.status_code != 200:
                raise RuntimeError('Unable to post telegram: %d %s' % (response.status_code, response.text))
            return response

    def get_channel_topic_with_id(self, chat_id=None):
        chat_id = chat_id or self.chat_id
        self.logger.info('Get chat pinned, id="%s"', chat_id)
        response = self.http_post(self.url('getChat'), data={'chat_id': chat_id}, headers=self.headers)
        return (
            response.json()['result']['pinned_message']['message_id'],
            response.json()['result']['pinned_message']['text'],
        )

    def get_channel_topic(self, chat_id=None):
        return self.get_channel_topic_with_id(chat_id)[1]

    def extract_login_from_topic(self, chat_id=None):
        topic = self.get_channel_topic(chat_id=chat_id or self.chat_id)
        return self.extract_login_from_line(topic)

    def extract_login_from_line(self, line):
        result = re.search('@\S+', line)
        if result:
            return result.group()[1:]  # skip `@`
        return None

    def update_duty_in_pinned_message(self, prefix, duty, chat_id=None):
        chat_id = chat_id or self.chat_id
        new_line = '%s: @%s' % (prefix, duty)
        pinned_message_id, current_pinned_message = self.get_channel_topic_with_id(chat_id=chat_id)
        current_pinned_message = current_pinned_message.split('\n')
        for line_index, line in enumerate(current_pinned_message):
            if line.startswith('dev '):
                del current_pinned_message[line_index]
            if line.startswith(prefix):
                current_duty = self.extract_login_from_line(line)
                if current_duty == duty:
                    self.logger.info('Current and New duty are same, chat_id=%s' % chat_id)
                    return None
                current_pinned_message[line_index] = new_line
                break
        else:
            current_pinned_message.append(new_line)
        self.edit_message(
            pinned_message_id,
            '\n'.join(sorted(current_pinned_message)),
            chat_id=chat_id,
        )
        self.send_message(u'Новый дежурный. ' + new_line, chat_id=chat_id)

    def edit_message(self, message_id, message, chat_id=None):
        chat_id = chat_id or self.chat_id
        self.logger.info('Edit message, chat_id="%s", message_id="%s"', chat_id, message_id)
        response = self.http_post(
            self.url('editMessageText'),
            data={
                'chat_id': chat_id,
                'message_id': message_id,
                'text': message,
            },
            headers=self.headers,
        )
        return response.json()['result']

    def send_message(self, message, chat_id=None):
        chat_id = chat_id or self.chat_id
        self.logger.info('Send message, chat_id="%s"', chat_id)
        response = self.http_post(
            self.url('sendMessage'),
            data={'chat_id': chat_id, 'text': message},
            headers=self.headers,
        )
        return response.json()['result']

    def pin_message(self, message_id, chat_id=None):
        chat_id = chat_id or self.chat_id
        self.logger.info('Set pinned message, chat_id="%s"', chat_id)
        response = self.http_post(
            self.url('pinChatMessage'),
            data={
                'chat_id': chat_id,
                'message_id': message_id,
                'disable_notification': True,
            },
            headers=self.headers,
        )
        return response.json()['result']

    def set_channel_topic(self, login, message_prefix=None, chat_id=None):
        message_prefix = message_prefix or 'Дежурный сегодня '
        chat_id = chat_id or self.chat_id
        result = self.send_message(message_prefix + "@{login}".format(login=login), chat_id=chat_id)
        return self.pin_message(result["message_id"], chat_id=chat_id)
