# -*- encoding: utf-8 -*-
import logging
import os
import requests

from utils import log_response


class TelegraphClient(object):
    """Документация: https://wiki.yandex-team.ru/VladimirPiskun/Telefonija-na-telegrafe/
    https://telegraph.yandex.net/api/v3/
    """
    URL = 'https://telegraph.yandex.net/api/v3/'

    def __init__(self, token):
        self.token = token
        self.logger = logging.getLogger('TelegraphClient')

    def _get_oauth_headers(self):
        return {
            'Authorization': 'OAuth %s' % self.token,
            'user-agent': 'disk.robot_switcher (GSID: %s)' % os.getenv('GSID', '').replace('\n', ' '),
        }

    def get_forward(self, robot_phone):
        self.logger.info('Get forward from="%s"', robot_phone)

        target_url = 'cucm/translation_pattern/pattern/%s' % robot_phone

        response = requests.get(TelegraphClient.URL + target_url, headers=self._get_oauth_headers())
        log_response(self.logger, response)
        if response.status_code != 200:
            raise RuntimeError('Unable to get forward: %d %s' % (response.status_code, response.text))

        result = response.json()
        if result['error']:
            raise RuntimeError('Telegraph returned error: %s' % str(result))

        forward_number = result['result']['calledPartyTransformationMask']
        self.logger.info('Forward is "%s"', forward_number)
        return forward_number

    def change_forward(self, robot_phone, worker_phone_number):
        worker_phone_number = worker_phone_number.encode('utf-8')
        worker_phone_number = worker_phone_number.translate(None, '() -')

        self.logger.info('Change forward: from="%s", to="%s"', robot_phone, worker_phone_number)

        target_url = 'cucm/translation_pattern/pattern/%s' % robot_phone
        response = requests.put(
            TelegraphClient.URL + target_url,
            json={"calledPartyTransformationMask": worker_phone_number},
            headers=self._get_oauth_headers())
        log_response(self.logger, response)
        if response.status_code != 200:
            raise RuntimeError('Unable to get forward: %d %s' % (response.status_code, response.text))

        result = response.json()
        if result['error']:
            raise RuntimeError('Telegraph returned error: %s' % str(result))

        self.logger.info('Successfully changed: "%s"', worker_phone_number)
        return
