# -*- encoding: utf-8 -*-
import datetime
import logging
import os
import requests
import time

from utils import log_response


class YandexABCClient(object):
    URL = 'https://abc-back.yandex-team.ru/api/v4/{endpoint}'

    def __init__(self, token):
        self.headers = {
            'Authorization': 'OAuth {}'.format(token),
            'user-agent': 'disk.robot_switcher (GSID: %s)' % os.getenv('GSID', '').replace('\n', ' '),
        }
        self.logger = logging.getLogger('YandexABCClient')

    def _get(self, endpoint, params=None, retries=3, sleep=10):
        url = self.URL.format(endpoint=endpoint)
        for attempt in range(retries):
            try:
                response = requests.get(url, params=params, headers=self.headers)
                response.raise_for_status()
                if not response.json():
                    raise Exception('empty response from abc')
            except Exception:
                self.logger.exception('get %s, try %s' % (endpoint, attempt))
            else:
                return response
            time.sleep(sleep)
        self.logger.error('cannot get %s after %s tries' % (endpoint, retries))
        return None

    def _get_duty_event(self, service_id, event_index=0, schedule__slug=None):
        self.logger.info('Get %s on duty' % service_id)
        url = 'services/%s/on_duty/?fields=replaces.person.login,person.login,id' % service_id
        if schedule__slug:
            url += '&schedule__slug=%s' % schedule__slug
        response = self._get(url)
        log_response(self.logger, response)
        return response.json()[event_index]

    def parse_duty_login(self, duty_event):
        if duty_event.get('replaces'):
            duty_event = duty_event['replaces'][0]
        return duty_event['person']['login']

    def get_duty(self, service_id, schedule__slug=None):
        return self.parse_duty_login(self._get_duty_event(service_id, schedule__slug=schedule__slug))

    def get_shifts(self, service_id, date_to, date_shift=7, schedule__slug=None):
        iso_format = '%Y-%m-%d'
        date_from = (
            datetime.datetime.strptime(date_to, iso_format) - datetime.timedelta(days=date_shift)
        ).strftime(iso_format)
        params = {
            'date_from': date_from,
            'date_to': date_to,
            'service': service_id,
            'fields': 'replaces.person.login,replaces.id,person.login,id',
        }
        if schedule__slug:
            params['schedule__slug'] = schedule__slug
        response = self._get('duty/shifts/', params=params)
        log_response(self.logger, response)
        return response.json()['results']

    def get_previous_duty(self, service_id, schedule__slug=None):
        current_duty = self._get_duty_event(service_id, schedule__slug=schedule__slug)
        shifts = self.get_shifts(service_id, current_duty['end'], schedule__slug=schedule__slug)
        if not shifts:
            return None
        previous_duty = shifts[0]
        for cursor_duty in shifts[1:]:
            cursor_duty_ids = {cursor_duty['id']} | {r['id'] for r in cursor_duty['replaces']}
            if current_duty['id'] in cursor_duty_ids:
                break
            previous_duty = cursor_duty
        if previous_duty:
            return self.parse_duty_login(previous_duty)
