# -*- encoding: utf-8 -*-
import datetime
import logging
import os
import requests
from lxml.html import fromstring


class YandexCalendarClient(object):
    URL = 'https://calendar.yandex-team.ru/{endpoint}'

    def __init__(self, token, tz):
        self.token = token
        self.tz_id = tz
        self.logger = logging.getLogger('YandexCalendarClientExport')
        self.headers = {
            'user-agent': 'disk.robot_switcher (GSID: %s)' % os.getenv('GSID', '').replace('\n', ' '),
        }

    def get_events(self, limit=1, start_ts=None):
        url = self.URL.format(endpoint='export/html.xml')
        query_params = {'private_token': self.token, 'tz_id': self.tz_id, 'limit': limit, 'lang': 'en'}
        if start_ts:
            query_params['start_ts'] = start_ts
        response = requests.get(url, params=query_params, headers=self.headers)
        calendar = fromstring(response.text)
        for event in calendar.find_class("b-content-event"):
            title = event.find("h1").text
            start = event.find_class("e-time")[0].find("span").text
            start_date = datetime.datetime.strptime(start, '%d %B %Y %H:%M').date()
            yield title, start_date

    def get_one_event(self, start_ts=None):
        events = self.get_events(limit=1, start_ts=start_ts)
        for event in events:
            return event
        else:
            if start_ts:
                dt = datetime.datetime.strptime(start_ts, '%Y-%m-%d').date()
            else:
                dt = datetime.datetime.now().date()
            return "No event", dt

    def get_current_event(self):
        self.logger.info('Get current event')
        return self.get_one_event()

    def get_previous_distinct_event(self, start_ts=None):
        if start_ts is None:
            start_ts = (datetime.datetime.now() - datetime.timedelta(days=14)).date().isoformat()
        events = list(self.get_events(limit=14, start_ts=start_ts))[::-1]
        current_event = self.get_current_event()
        current_index = events.index(current_event)
        for event in events[current_index + 1:]:
            if event[0] != current_event[0]:
                return event
        return None, None

    @staticmethod
    def is_today(date):
        now = datetime.datetime.now()
        return date <= now.date() and YandexCalendarClient.is_day()

    @staticmethod
    def is_day():
        now = datetime.datetime.now()
        return now.hour > 10 and now.hour < 22
