# -*- encoding: utf-8 -*-
import logging
import os
import requests

from utils import log_response


class YandexStaffClient(object):
    URL = 'https://staff-api.yandex-team.ru/v3/{endpoint}'

    def __init__(self, token):
        self.headers = {
            'Authorization': 'OAuth {}'.format(token),
            'user-agent': 'disk.robot_switcher (GSID: %s)' % os.getenv('GSID', '').replace('\n', ' '),
        }
        self.logger = logging.getLogger('YandexStaffClient')

    @staticmethod
    def get_call_number(person):
        work_phone = person['work_phone']
        if work_phone:
            return "55%s" % work_phone
        for phone in person['phones']:
            if phone['type'] == 'mobile':
                return phone['number']
        return None

    def _get(self, endpoint, params, retries=3):
        url = self.URL.format(endpoint=endpoint)
        for attempt in range(retries):
            try:
                response = requests.get(url, params=params, headers=self.headers)
                response.raise_for_status()
            except Exception:
                self.logger.exception('get %s, try %s' % (endpoint, attempt))
            else:
                return response
        self.logger.error('cannot get %s after %s tries' % (endpoint, retries))
        return None

    def get_worker_phone_number(self, login):
        self.logger.info('Get %s phone number' % login)
        query_params = {'login': login, '_fields': 'work_phone,phones'}
        response = self._get('persons', query_params)
        log_response(self.logger, response)
        person = response.json()['result'][0]
        return self.get_call_number(person)

    def get_group_members(self, department):
        self.logger.info('Get %s members' % department)
        query_params = {
            'groups.group.url': department,
            'official.is_dismissed': 'false',
            '_limit': '150',
            '_fields': 'login,work_phone,phones.number,phones.type'
        }
        response = self._get('persons', query_params)
        log_response(self.logger, response)
        return response.json()['result']

    def get_telegram_username(self, login):
        self.logger.info('Get %s telegram' % login)
        query_params = {
            'login': login,
            '_fields': 'accounts.value,accounts.type'
        }
        response = self._get('persons', query_params)
        log_response(self.logger, response)
        accounts = response.json()['result'][0]['accounts']
        telegram_accounts = filter(lambda a: a['type'] == 'telegram', accounts)
        if not telegram_accounts:
            return None
        return telegram_accounts[0]['value']

    def get_login_by_telegram(self, username):
        self.logger.info('Get login by telegram %s' % username)
        query_params = {
            '_query': 'accounts==match({"type":"telegram","value_lower":"%s"})' % username.lower(),
            '_fields': 'login'
        }
        response = self._get('persons', query_params)
        log_response(self.logger, response)
        result = response.json()['result']
        if not result:
            return None
        return result[0]['login']
