#!/bin/bash

BACKUP_DIR=/u0/backup
BACKUP_DAYS=8
DT_HOURS=12
DT_DESCRIPTION="MongoDB backup on $(hostname)"
DT_NAMESPACE=disk
DEFRAG_FLAG="/var/tmp/mongo-defrag-in-progress"
JUGGLER_AUTH="$(cat /etc/yandex/juggler.oauth)"
KEY_FILE="/etc/yandex/backup_mongo_aes.key"
ENDPOINT_URL="https://s3.mds.yandex.net"
AWS_CMD="aws --endpoint-url=${ENDPOINT_URL} s3"
export AWS_CONFIG_FILE="/etc/yandex/aws.config"


[ ! -d $BACKUP_DIR ] && mkdir -p $BACKUP_DIR
function log {
    echo -n $(date "+%F %H:%M:%S") ; echo " " $@;
}

function downtime_host {
    hostname=$1
    log "Trying to downtime $hostname for $DT_HOURS hours"
    endtime=$(date -d "+$DT_HOURS hours" +%s)
    downtime_id=$(curl -ks \
        -H "Authorization: OAuth $JUGGLER_AUTH" \
        -H 'Content-Type: application/json' \
        -d "{\"end_time\":$endtime, \
            \"description\": \"$DT_DESCRIPTION\", \
            \"filters\": [{\"host\": \"$hostname\", \"namespace\": \"$DT_NAMESPACE\"}]}" \
        https://juggler-api.search.yandex.net/v2/downtimes/set_downtimes | \
        grep downtime_id | awk '{print $2}' | cut -d\" -f 2)
    return $?
}

function remove_downtime {
    hostname=$1
    log "Trying to remove $hostname downtime"
    curl -ks \
        -H "Authorization: OAuth $JUGGLER_AUTH" \
        -H 'Content-Type: application/json' \
        -d "{\"downtime_ids\": [\"${downtime_id}\"]}" \
        http://juggler-api.search.yandex.net/v2/downtimes/remove_downtimes
    return $?
}

function error_exit {
    log "Catched error, something was gone a wrong way. Cleanup"
    lxc-attach -n $name -- /etc/init.d/mongodb start
    log "Backup failed!"
    exit -1
}

function s3_upload {
    HTTP_PROXY="$(curl -s ${ENDPOINT_URL}/hostname):4080"
    $AWS_CMD cp "$1" s3://disk-backup-mongo/backup/$2/ || (log "s3 upload for $2 failed" && return 1)
}

function s3_clean {
    host=$1
    for backup_file in $($AWS_CMD ls s3://disk-backup-mongo/backup/${host}/ | awk '{print $NF}' | sort); do
        # shard.dump.${iso_date}.tar.gz
        dump_date=$(date -d "$(echo $backup_file | cut -d. -f3)" +%s)
        older_than=$(date -d "-${BACKUP_DAYS} days" +%s)
        if [ $dump_date -lt $older_than ]; then
            $AWS_CMD rm "s3://disk-backup-mongo/backup/${host}/$backup_file"
        else
            break
        fi
    done
}

function backup_mongo {
    host=$1
    name=$host
    rootfs=$(awk '/root:/ {print $2}' /etc/lxctl/${host}.yaml | tr -d " '")/rootfs
    socket=$(readlink -f ${rootfs}/tmp/mongodb*27018*.sock)
    log "Working on $name"
    if [ ! -S $socket ] ; then
        log "Mongodb socket was not found (check $socket), will skip $name"
        return
    fi

    if [ -f "${rootfs}/root/.password" ] ; then
        password=$(cat ${rootfs}/root/.password)
        mongo_command="${rootfs}/usr/bin/mongo --host $socket -p${password} -u admin --authenticationDatabase admin --quiet --eval"
    else
        mongo_command="/usr/bin/mongo --host $socket --quiet --eval"
    fi
    
    dbpath=${rootfs}$($mongo_command "opts=db.serverCmdLineOpts().parsed; print(opts.hasOwnProperty('storage')?opts.storage.dbPath:opts.dbpath)")
    iso_date=$(date -Idate)
    host_backup_dir=${BACKUP_DIR}/${host}/${iso_date}

    # Check defragmentation flag
    if [ -f "$rootfs/${DEFRAG_FLAG}" ]; then
        log "Defrag flag exists (${DEFRAG_FLAG}), will skip $name"
        return
    fi

    # Check replication state
    mongo_state=$($mongo_command 'printjson(rs.status().myState)')
    if [ "$mongo_state" == "undefined"  ] ; then
        log "Undefined server status: check replication settings. Will skip $name"
        return
    fi

    if [ $mongo_state -ne 2 ] ; then
        log "Wrong server state: $mongo_state. State 2 (SECONDARY) was expected. Will skip $name"
        return
    fi


    [ ! -d $host_backup_dir ] && mkdir -p $host_backup_dir
    downtime_host $host

    lxc-attach -n $name -- /usr/sbin/service mongodb stop
    log "mongodb in $host was stopped, will sleep for 2 minutes"
    sleep 120

    log "Taking snapshot (encrypted) for $name"
    dump_file="$host_backup_dir/shard.dump.${iso_date}.tar.gz.enc"
    (cd $dbpath ; tar cvf - . | pigz -c | openssl aes-256-cbc -kfile ${KEY_FILE} -salt -out "${dump_file}")

    # starting mongodb
    lxc-attach -n $name -- /usr/sbin/service mongodb start
    log "mongodb in $name was started"

    s3_upload "$dump_file" $host || s3_upload "$dump_file" $host || s3_upload "$dump_file" $host
    s3_clean $host
    rm "$dump_file"
}

trap error_exit ERR
set -e

log "Cleaning up old backups (cause s3)"
find $BACKUP_DIR -type f -print -delete

for host in $(/usr/bin/lxctl list | awk '$3=="running" || $3=="b" {print $1}' | grep hidden); do
    if [ -f /etc/lxctl/${host}.yaml ] ; then
        backup_mongo $host
    fi
done
