#!/usr/bin/env python

import os
import sys
import requests
import socket
from collections import OrderedDict, namedtuple

Host = namedtuple('Host', ['name', 'address'])

CONDUCTOR_GROUPS = [
    'elliptics-storage',
    'disk_uploader',
    'disk_downloader_central'
]

STATIC_HOSTS = [
    'an.yandex.ru', 'api.disk.yandex.net', 'awaps.yandex.ru', 'bar-navig.yandex.ru',
    'bs-meta.yandex.ru', 'certificate-pin.yandex.net', 'clck.yandex.ru', 'cloud-api.yandex.net',
    'cloud-api.yandex.ru', 'cloud-api.yandex.ua', 'contact2.yandex.com.tr', 'disk.yandex.com',
    'disk.yandex.com.tr', 'disk.yandex.ru', 'disk.yandex.ua', 'downloader.disk.yandex.ru', 'downloader.disk.yandex.ua',
    'feedback2.yandex.com', 'feedback2.yandex.ru', 'feedback2.yandex.ua',
    'flv.video.yandex.ru', 'fotki.yandex.ru', 'front.disk.yandex.ru',
    'help.yandex.com', 'help.yandex.com.tr', 'help.yandex.ru', 'help.yandex.ua', 'img.fotki.yandex.ru',
    'legal.rostaxi.org', 'legal.taxitax.org', 'legal.yandex.com', 'legal.yandex.com.tr', 'legal.yandex.ru',
    'legal.yandex.ua', 'log.disk.yandex.net', 'mc.yandex.ru', 'mobileproxy.passport.yandex.net',
    'oauth.mobile.yandex.net', 'oauth.rostaxi.org', 'oauth.taxitax.org', 'oauth.yandex.com',
    'oauth.yandex.com.tr', 'oauth.yandex.ru', 'passport.yandex.com', 'passport.yandex.com.tr',
    'passport.yandex.ru', 'passport.yandex.ua', 'proxy.video.yandex.net', 'push.xmpp.yandex.ru',
    'push.yandex.ru', 'registrator.mobile.yandex.net', 'registrator.rostaxi.org', 'registrator.taxitax.org',
    'soft.export.yandex.ru', 'streaming.video.yandex.ru', 'streaming.disk.yandex.net', 'sync-keys.disk.yandex.net',
    'sync.disk.yandex.net', 'video.yandex.com', 'video.yandex.com.tr', 'video.yandex.ru', 'video.yandex.ua',
    'webdav.tst.yandex.ru', 'webdav.yandex.com', 'webdav.yandex.com.tr', 'webdav.yandex.ru',
    'webdav.yandex.ua', 'xmpp.disk.yandex.net', 'yandex.com', 'yandex.com.tr', 'yandex.ru', 'yandex.st',
    'yandex.ua', 'yapic.yandex.net', 'yardim.yandex.com.tr',
]


class ConfigUpdater:
    HAPROXY_TEMPLATE = """
global
    daemon
    maxconn 20000
    user haproxy
    group haproxy
    stats socket /var/run/haproxy.sock mode 0600 level admin
    log /dev/log  local0 debug
    pidfile /var/run/haproxy.pid
    spread-checks 5

defaults
    maxconn 20000
    log global
    mode http
    option httplog
    option abortonclose
    option http-server-close
    option persist
    option accept-invalid-http-response

    timeout connect 20s
    timeout server 3000s
    timeout client 3000s
    timeout check 10s
    retries 3

frontend aws-sni-frontend
#  bind 172.22.0.57:443
#  bind 172.22.0.58:443
  bind 5.45.202.26:443
  bind 5.45.202.27:443
  bind 5.45.202.28:443
  bind 5.45.202.29:443

  mode tcp
  log global
  option tcplog
  no option http-server-close

  tcp-request inspect-delay 5s
  tcp-request content accept               if { req_ssl_hello_type 1 }

  use_backend aws-sni-backend              if !{ req_ssl_sni -i blah }
  default_backend aws-deadend-sni

backend aws-sni-backend
  log global
  option tcplog
  mode tcp
  no option http-server-close
  no option accept-invalid-http-response

@HOSTS@

backend aws-deadend
  mode http
  log global
  option httplog

backend aws-deadend-sni
  mode tcp
  log global
  option tcplog
  no option accept-invalid-http-response
  no option http-server-close
""".lstrip()

    SQUID_TEMPLATE = """
@ACLS@

acl ssl_ports port 443
acl safe_ports port 80
acl safe_ports port 443
acl CONNECT method CONNECT

http_access deny !safe_ports
http_access deny CONNECT !ssl_ports

http_access allow allow_yandex_nets
http_access deny all

http_port 5.45.202.22:443
http_port 5.45.202.23:443
http_port 5.45.202.24:443
http_port 5.45.202.25:443

cache_mem 20 GB
maximum_object_size_in_memory 100 MB
maximum_object_size 1000 MB
memory_replacement_policy lru
cache_replacement_policy lru

cache_dir ufs /var/spool/squid3 751016 16 256
store_dir_select_algorithm least-load
max_open_disk_fds 0

{% raw -%}
logformat squid %{%Y-%m-%d %H:%M:%S}tl,%03tu %6tr %>a %Ss/%03>Hs %<st %rm %ru %un %Sh/%<A %mt
access_log /var/log/squid3/access.log squid

logformat tskv tskv	\\
{%- endraw %}
	tskv_format={{ pillar.get('squid-access-log-tskv-format', 'ydisk-proxy-access-log') }}	\\
	host={{grains.get('fqdn')}}	\\
{%- raw %}
	timestamp=%{%Y-%m-%dT%H:%M:%S}tl	timezone=%{%z}tl	\\
	unixtime_ms=%ts.%03tu	\\
	request_time=%tr	upstream_response_time=%<tt	ip=%>a	\\
	upstream=%<A	client_size=%<sH	upstream_size=%<sS	\\
	bytes_sent=%<st	request=%ru	content_type=%mt	\\
	status=%>Hs	upstream_status=%<Hs	user=%[un	\\
	squid_hier_status=%Sh	squid_request_status=%Ss	\\
	protocol=%rv	method=%rm	request_length=%>st
{% endraw -%}
access_log /var/log/squid3/access-tskv.log tskv

pid_filename /var/run/squid3.pid

buffered_logs on
cache_log /var/log/squid3/cache.log

coredump_dir /var/spool/squid3

connect_timeout 15 seconds
read_timeout 60 minutes
request_timeout 60 minutes

shutdown_lifetime 30 seconds
cache_effective_user proxy
cache_effective_group proxy

umask 027

client_db on
retry_on_error off
""".lstrip()

    def __init__(self, conductor_groups=None, static_hosts=None):
        self.conductor_hosts = conductor_groups or CONDUCTOR_GROUPS
        self.static_hosts = static_hosts or STATIC_HOSTS
        self.hosts = self._get_hosts()

    def _get_hosts(self):
        hosts_dict = OrderedDict()
        hosts_dict['static'] = []

        def append(hostname, group):
            addrs = self.getaddr(hostname)
            if addrs:
                hosts_dict[group].append(Host(name=hostname, address=addrs))

        for hostname in self.static_hosts:
            append(hostname, 'static')
        for group in self.conductor_hosts:
            hosts_dict[group] = []
            for hostname in self.get_conductor_hosts(group):
                append(hostname, group)
        return hosts_dict

    @staticmethod
    def get_conductor_hosts(group_name):
        url = "http://c.yandex-team.ru/api/groups2hosts/%s?format=json" % group_name
        r = requests.get(url)
        return set(h["fqdn"] for h in r.json())

    @staticmethod
    def getaddr(hostname):
        try:
            addrs = socket.getaddrinfo(hostname, None)
        except socket.gaierror:
            return None
        addrs = set([_[4][0] for _ in addrs])
        return addrs

    def prepare_haproxy(self):
        hosts_replace = []
        for group, hosts in self.hosts.items():
            hosts_replace.append("#  %s section" % group)
            for host in sorted(hosts, key=lambda x: x.name):
                context = {'host': host.name}
                hosts_replace.append("  use-server %(host)s if { req.ssl_sni -m str %(host)s }" % context)
                hosts_replace.append(
                    "  server %(host)s %(host)s:443 check inter 10s fastinter 2s downinter 2s fall 1800" % context)
                hosts_replace.append("")
        result = self.HAPROXY_TEMPLATE.replace("@HOSTS@", "\n".join(hosts_replace))
        return result

    def prepare_squid(self):
        replace = []
        ips = []
        for group, hosts in self.hosts.items():
            for host in hosts:
                ips.extend(host.address)
        for ip in sorted(set(ips)):
            replace.append("acl allow_yandex_nets dst %s" % ip)
        result = self.SQUID_TEMPLATE.replace("@ACLS@", "\n".join(replace))
        return result


if __name__ == "__main__":
    c = ConfigUpdater()
    haproxy = c.prepare_haproxy()
    squid = c.prepare_squid()
    if len(sys.argv) > 1 and sys.argv[1] == '--dry':
        print('=' * 6 + 'haproxy' + '=' * 6)
        print(haproxy)

        print('=' * 6 + 'squid' + '=' * 6)
        print(squid)
    else:
        dst = os.path.dirname(os.path.abspath(__file__))
        with open(os.path.join(dst, 'haproxy/haproxy.cfg'), 'w') as f:
            f.write(haproxy)
        with open(os.path.join(dst, '../../../units/squid3/files/etc/squid3/squid.conf-disk_lb'), 'w') as f:
            f.write(squid)
