#!/bin/bash

jctl_hosts_backup=("jmon01e.media.yandex.net" "jmon01g.media.yandex.net" "jmon01h.media.yandex.net")
jctl_hosts=( $(curl -ks 'http://c.yandex-team.ru/api-cached/groups2hosts/media-jmon') )
jctl_hosts="${jctl_hosts:-$jctl_hosts_backup}"

backup_dir=/u0
backup_days=2
dt_hours=8
dt_sync_hours=3
dt_description="MongoDB backup"
defrag_flag="/var/tmp/mongo-defrag-in-progress"


function log {
    echo -n $(date "+%F %H:%M:%S") ; echo " " $@; 
}

function downtime_host
{
    ok=0
    hostname=$1
    dt_time=$2
    dt_desc=$3
    for host in ${jctl_hosts[*]}; do 
        log "Trying to downtime $hostname for $dt_time hours at $host (${dt_desc})"
        curl -ks "http://${host}:8998/api/downtimes/set_downtime?do=1&object_name=$hostname&end_time=%2B${dt_time}hours&description=${dt_desc}"; echo
        ok=$?
        if [[ ${ok} == 0 ]]; then 
            break
        fi
    done
    return $ok
}

function remove_downtime {
    ok=0
    hostname=$1
    for host in ${jctl_hosts[*]}; do 
        log "Trying to remove $hostname downtime at $host"
        curl -ks "http://${host}:8998/api/downtimes/remove_downtime?do=1&object_name=$hostname"; echo
        ok=$?
        if [[ ${ok} == 0 ]]; then 
            break
        fi
    done
    return $ok
}

function error_exit {
    log "Backup failed!"
    exit -1
}

trap error_exit ERR
set -e


[ ! -d $backup_dir ] && mkdir -p $backup_dir

log "Cleaning up old backups (older than ${backup_days} days)"
find $backup_dir -type f -ctime +${backup_days} -delete

name=$(hostname -f)
host=${name}

socket=$(readlink -f /tmp/mongodb*27018*.sock)
log "Working on $name"
if [ -S $socket ] ; then
    dbpath=$(mongo --host $socket --quiet --eval "opts=db.serverCmdLineOpts().parsed; print(opts.hasOwnProperty('storage')?opts.storage.dbPath:opts.dbpath)")
    host_backup_file="${backup_dir}/$(date +%Y%m%d).gz"
    # Check defragmentation flag
    if [ -f "${defrag_flag}" ]; then
        log "Defrag flag exists (${defrag_flag}), will skip $name"
        exit 1
    fi  

    # Check replication state
    mongo_state=`mongo --quiet --host $socket --eval 'printjson(rs.status().myState)'`
    if [ "$mongo_state" == "undefined"  ] ; then
        log "Undefined server status: check replication settings. Will skip $name"
        exit 1
    fi

    if [ $mongo_state -ne 2 ] ; then
        log "Wrong server state: $mongo_state. State 2 (SECONDARY) was expected. Will skip $name"
        exit 1
    fi


    [ ! -d $host_backup_dir ] && mkdir -p $host_backup_dir
    downtime_host $host ${dt_hours} "Backup hidden node"
    #mongo --quiet --host $socket --eval 'printjson(db.fsyncLock())'
        

    service mongodb stop
    log "mongodb in $host was stopped, will sleep for 2 minutes"
    sleep 120

    log "Taking snapshot for $name"
    (cd $dbpath ; tar cvf - . | pigz -c > "$host_backup_file" )
            

    # starting mongodb
    service mongodb start 
    #mongo --quiet --host $socket --eval 'printjson(db.fsyncUnlock())'
    log "mongodb in $name was started"

    remove_downtime $host

    downtime_host $host ${dt_sync_hours} "Replica sync after backup"

    echo 
    echo
else
    log "Mongodb socket was not found (check $socket), will skip $name"
fi
