#!/usr/bin/env python

import sys
from collections import Counter, defaultdict

ENVOY_CLUSTER_PREFIX = 'envoy_cluster_'
HISTOGRAM_SUFFIX = '_bucket'
STATUSES_SUFFIX = '_rq_xx'
RETRY_SUFFIX = '_rq_retry'
OTHER_METRICS = {
    'upstream_flow_control',
    'upstream_cx',
    'upstream_rq',
}

all_timings = defaultdict(Counter)
all_stats = Counter()


def contains_one_of(item, container):
    for element in container:
        if element in item:
            return True
    return False


def fix_prometheus_histogram(timings):
    histogram = [[0, 0]]
    counted = 0
    for key, value in sorted(timings.items()):
        histogram[-1][1] = value - counted
        histogram.append([key, 0])
        counted = value
    return histogram


def get_prometheus_value(line, key):
    key = key + '="'
    value_index = line.find(key) + len(key)
    return line[value_index:value_index + line[value_index:].find('"')]


for line in sys.stdin:
    if line.startswith('#') or not line.strip() or not line.startswith(ENVOY_CLUSTER_PREFIX):
        continue

    metric_name, remain = line.strip().split('{')
    envoy_cluster_name = get_prometheus_value(remain, 'envoy_cluster_name')
    if metric_name.startswith(ENVOY_CLUSTER_PREFIX) and envoy_cluster_name:
        metric_name = ENVOY_CLUSTER_PREFIX + envoy_cluster_name + '_' + metric_name[len(ENVOY_CLUSTER_PREFIX):]

    value = remain.split()[-1]

    if metric_name.endswith(HISTOGRAM_SUFFIX):
        metric_name = metric_name[:-len(HISTOGRAM_SUFFIX)] + '_timings'
        le_value = get_prometheus_value(remain, 'le')
        if le_value == '+Inf':
            continue
        all_timings[metric_name][float(le_value)] = int(value)
    elif metric_name.endswith(STATUSES_SUFFIX):
        envoy_status_code = get_prometheus_value(remain, 'envoy_response_code_class')
        metric_name = metric_name[:-len(STATUSES_SUFFIX)] + '_' + envoy_status_code + 'xx'
        all_stats[metric_name] = value
    elif metric_name.endswith(RETRY_SUFFIX) or contains_one_of(metric_name, OTHER_METRICS):
        all_stats[metric_name] = value


for key, timings in sorted(all_timings.items()):
    if timings:
        timings = fix_prometheus_histogram(timings)
        packed_timings = map(lambda t: "%s@%s" % (t[0], t[1]), timings)
        print("@%s %s" % (key, ' '.join(packed_timings)))

for key, value in sorted(all_stats.items()):
    print("%s %s" % (key, value))
