# -*- coding: utf-8 -*-
"""
A module that adds data to the Pillar structure retrieved by an https request
to DBaaS Internal API


Configuring the dbaas_pillar ext_pillar
=======================================

Set the following Salt config to setup dbaas_pillar as external pillar source:

.. code-block:: yaml

  ext_pillar:
    - dbaas_pillar:
        url: https://internal-api-host/api/v1.0/config/
        access_id: <Internal API id>
        access_secret: <Internal API secret>
        api_pub_key: <internal api public key>
        salt_sec_key: <salt secret key>
        target_pillar_id: <optional target id>

"""
from __future__ import absolute_import

import collections
import json
import logging
import os

import requests
from nacl.encoding import URLSafeBase64Encoder as encoder
from nacl.public import Box, PrivateKey, PublicKey


def _decrypt(secret_key, public_key, data):
    """
    Decrypt data using secret key and public key
    """
    version = data['encryption_version']
    if version == 1:
        box = Box(
            PrivateKey(secret_key.encode('utf-8'), encoder),
            PublicKey(public_key.encode('utf-8'), encoder))
        return box.decrypt(encoder.decode(
            data['data'].encode('utf-8'))).decode('utf-8')
    else:
        raise RuntimeError('Unexpected encryption version: %d' % version)


def _rec_decrypt(inp_dict, sec_key, pub_key):
    """
    Recursive dictionary decryption
    """

    def _update(ret, update):
        for key, value in update.items():
            if isinstance(value, collections.Mapping):
                if 'encryption_version' in value and 'data' in value:
                    ret[key] = _decrypt(sec_key, pub_key, value)
                else:
                    ret[key] = _update({}, value)
            else:
                ret[key] = update[key]

        return ret

    result = {}
    _update(result, inp_dict)
    return result


def ext_pillar(  # pylint: disable=too-many-arguments
        minion_id, pillar, urls, access_id, access_secret, api_pub_key,
        salt_sec_key):
    """
    Read pillar data from HTTPS response
    """
    log = logging.getLogger(__name__)
    cache_path = os.path.join('/dev/shm', 'dbaas-%s' % minion_id)

    request_headers = {
        'Accept': 'application/json',
        'Access-Id': access_id,
        'Access-Secret': access_secret,
        'Content-Type': 'application/json',
    }

    if pillar.get('data', {}).get('runlist', []):
        return {}

    request_params = None
    if 'target-pillar-id' in pillar:
        request_params = {'target-pillar-id': pillar['target-pillar-id']}

    try:
        for url in urls:
            response = requests.get(
                url + minion_id,
                params=request_params,
                headers=request_headers,
                verify=False,
            ).json()
            if response:
                break
        try:
            with open(cache_path, 'w') as cache_file:
                cache_file.write(json.dumps(response))
        except Exception:
            pass
    except Exception as exc:
        log.exception(exc)
        response = {}
        try:
            with open(cache_path) as cache_file:
                response = json.loads(cache_file.read())
        except Exception:
            pass

    try:
        return _rec_decrypt(response, salt_sec_key, api_pub_key)
    except Exception as exc:
        log.exception(exc)
        return {}
