#!/usr/bin/env python

import json
import os

CACHE_FILE_NAME = '/tmp/.grains_mongodb.cache'

MONGO_STATES = {
    0: 'STARTUP',
    1: 'PRIMARY',
    2: 'SECONDARY',
    3: 'RECOVERING',
    5: 'STARTUP2',
    6: 'UNKNOWN',
    7: 'ARBITER',
    8: 'DOWN',
    9: 'ROLLBACK',
    10: 'REMOVED',
}


def _read_cache(filename):
    if os.path.exists(filename):
        with open(filename, 'r') as cache:
            try:
                return json.loads(cache.read())
            except (KeyError, ValueError):
                return {}


def _write_cache(filename, data):
    with open(filename, 'w') as cache:
        cache.write(json.dumps(data))


def mongodb():
    data = {}
    try:
        import pymongo
        user = 'monitor'
        host = 'localhost'
        port = 27018
        dbname = 'admin'
        password = ''
        timeout_ms = 5 * 1000
        rs_status = None

        with open(os.path.expanduser("~/.mongopass")) as mongopass:
            for line in mongopass:
                tokens = line.rstrip().split(':')
                if tokens[3] == user:
                    host, port, dbname, user, password = tokens
                    break

        options = dict((opt, timeout_ms) for opt in ('connectTimeoutMS',
                                                     'socketTimeoutMS',
                                                     'serverSelectionTimeoutMS'))

        try:
            conn = pymongo.MongoClient(host='localhost',
                                       port=int(port),
                                       **options)
            server_info = conn.server_info()
            data['version'] = server_info['version']

            _write_cache(CACHE_FILE_NAME, data)

            rs_status = conn.admin.command('replSetGetStatus')
            data['replset_initialized'] = True
            data['auth_enabled'] = False

        except pymongo.errors.OperationFailure as exc:
            if 'no replset config has been received' in str(exc):
                data['replset_initialized'] = False

            elif 'not authorized on admin' in str(exc):
                data['auth_enabled'] = True
                data['replset_initialized'] = True
                try:
                    conn = pymongo.MongoClient(host=host,
                                               port=int(port),
                                               **options)
                    mdb = pymongo.database.Database(conn, dbname)
                    mdb.authenticate(user, password, source=dbname)

                    rs_status = conn.admin.command('replSetGetStatus')

                except pymongo.errors.OperationFailure as exc:
                    if 'Authentication failed' in str(exc):
                        raise

        if rs_status:
            data['my_state'] = rs_status['myState']
            data['state_str'] = MONGO_STATES[rs_status['myState']]

    except Exception:
        data = _read_cache(CACHE_FILE_NAME)
    return {'mongodb': {'mongod': data}}


if __name__ == '__main__':
    from pprint import pprint
    pprint(mongodb())
