# -*- coding: utf-8 -*-
"""
Redis module for salt
"""

from __future__ import absolute_import, print_function, unicode_literals

import logging
import socket

from salt.ext import six

try:
    from redis import StrictRedis
    from redis.exceptions import AuthenticationError, ResponseError
    from redis.sentinel import Sentinel
    HAS_REDIS_LIB = True
except ImportError:
    HAS_REDIS_LIB = False

LOG = logging.getLogger(__name__)


def __virtual__():
    """
    We always return True here to allow using calls to module
    in plain states
    """
    return True


def _get_config_option(option, config_path):
    """
    Get full value of option from config file.
    We expect config generated by our salt component here.
    """
    string_start = '{option} '.format(option=option)
    try:
        with open(config_path) as config:
            for line in config:
                if line.startswith(string_start):
                    stripped = line.split(' ', 1)[1].strip()
                    # Workaround for quotted config options
                    # (created by config rewrite command)
                    if stripped.startswith('"') and stripped.endswith('"'):
                        return stripped[1:-1]
                    return stripped
    except Exception as exc:
        LOG.error('Unable to get %s from %s: %s', option, config_path,
                  repr(exc))


def _get_connection(password, port, config_path, host='localhost'):
    """
    Get connection with redis.
    If auth fails with supplied password
    we fall back to requirepass value from config.
    This is dirty but allows password change via pillar.
    """
    try:
        conn = StrictRedis(host=host, port=port, db=0, password=password)
        conn.ping()
        return conn
    except AuthenticationError as exc:
        LOG.warning('Connection with initially supplied password failed: %s',
                    exc)
        fallback_password = _get_config_option('requirepass', config_path)
        conn = StrictRedis(
            host='localhost', port=port, db=0, password=fallback_password)
        conn.ping()
        return conn


def _get_config_cmd_name(config_path):
    """
    Get renamed CONFIG command from config.
    We rename this command to avoid changing config options by
    connected clients.
    """
    rename = _get_config_option('rename-command CONFIG', config_path)
    if rename:
        return rename.replace('CONFIG', '', 1).lstrip()
    return 'CONFIG'


def _ensure_option(conn, config_cmd_name, test, option, value):
    """
    Ensure that option is set to value.
    """
    try:
        ret = conn.execute_command(config_cmd_name, 'get', option)
    except Exception as exc:
        LOG.debug('Unable to get %s option value: %s', option, repr(exc))
        ret = None
    if not ret:
        # Unknown config option. Skip
        return
    _, current_value = ret
    str_val = str(value) if not isinstance(value, six.string_types) else value
    if current_value != str_val:
        if test:
            return value
        try:
            ret = conn.execute_command(config_cmd_name, 'set', option, str_val)
            if ret == 'OK':
                return value
        except ResponseError as exc:
            LOG.warning('Changing config %s failed: %s', option, repr(exc))


def set_options(options,
                password=None,
                port=6379,
                test=False,
                config_path='/etc/redis/redis-main.conf'):
    """
    Set config options with config rewrite in the end.
    Returns dict of changed values.
    """
    if not HAS_REDIS_LIB:
        raise RuntimeError('Unable to import redis module')
    changed = {}
    config_cmd_name = _get_config_cmd_name(config_path)
    conn = _get_connection(
        password=password, port=port, config_path=config_path)
    for option, value in options.items():
        changed[option] = _ensure_option(conn, config_cmd_name, test, option,
                                         value)

    if any(changed.values()):
        if not test:
            conn.execute_command(config_cmd_name, 'rewrite')
        return dict((k, v) for k, v in changed.items() if v)


def _get_hostname(address):
    """
    Simple wrapper for socket.gethostbyaddr
    """
    try:
        return socket.gethostbyaddr(address)[0]
    except socket.herror as exc:
        LOG.warning('Unable to resolve %s: %s', address, repr(exc))
        return address


def _get_sentinel_master(hosts, port, master_name):
    """
    Get master ip from connection with sentinels.
    Unfortunately sentinels know nothing about auth atm
    (See https://github.com/antirez/redis/pull/3329 for details).
    """
    if not HAS_REDIS_LIB:
        raise RuntimeError('Unable to import redis module')
    conns = []
    for host in hosts:
        conns.append((host, port))
    sentinel_conn = Sentinel(conns, socket_timeout=1)

    host, _ = sentinel_conn.discover_master(master_name)

    # We need hostname here
    return _get_hostname(host)


def get_master(hosts, port, master_name):
    try:
        return _get_sentinel_master(hosts, port, master_name)
    except Exception as exc:
        LOG.debug('Unable to get redis master: %s', repr(exc))
