# -*- coding: utf-8 -*-
'''
Module to provide MongoDB functionality to Salt

:configuration: This module uses PyMongo, and accepts configuration details as
    parameters as well as configuration settings::

        mongodb.host: 'localhost'
        mongodb.port: 27017
        mongodb.user: ''
        mongodb.password: ''

    This data can also be passed into pillar. Options passed into opts will
    overwrite options passed into pillar.
'''
from __future__ import absolute_import, print_function, unicode_literals

# Import python libs
import logging
import re
import sys
import time

# Import salt libs
import json

#> dbaas
#> from salt.utils.versions import LooseVersion as _LooseVersion
from salt.exceptions import get_error_message as _get_error_message


# Import third party libs
from salt.ext import six
try:
    import pymongo
    HAS_MONGODB = True
except ImportError:
    HAS_MONGODB = False

log = logging.getLogger(__name__)

MEMBER_STR_STATES = {
    'STARTUP': 0,
    'PRIMARY': 1,
    'SECONDARY': 2,
    'RECOVERING': 3,
    'STARTUP2': 5,
    'UNKNOWN': 6,
    'ARBITER': 7,
    'DOWN': 8,
    'ROLLBACK': 9,
    'REMOVED': 10,
}

MEMBER_INT_STATES = dict((v, k) for k, v in MEMBER_STR_STATES.items())


MEMBER_ERROR_STATES = {
    'UNKNOWN': 6,
    'DOWN': 8,
    'ROLLBACK': 9,
    'REMOVED': 10,
}

RS_CHANGE_WAIT_SECS = 10
SECONDARY_CATCH_UP_PERIOD_SECS = 10
DEFAULT_TIMEOUT_SECS = 5


def __virtual__():
    '''
    Only load this module if pymongo is installed
    '''
    if not HAS_MONGODB:
        return (False, 'The mongodb execution module cannot be loaded: the pymongo library is not available.')

    return 'mongodb'


def _connect(user=None, password=None, host=None, port=None, database=None, authdb=None, options=None):
    '''
    Returns a tuple of (user, host, port) with config, pillar, or default
    values assigned to missing values.
    '''
    if host is None:
        host = 'localhost'
    if port is None:
        port = 27018
    if database is None:
        database = 'admin'
    if authdb is None:
        authdb = database
    if options is None:
        options = {}

    try:
        log.debug('Trying to connect: %s:%s %s', host, port, options)
        conn = pymongo.MongoClient(host=host, port=port, **options)
        mdb = pymongo.database.Database(conn, database)
        if user and password:
            log.debug('Auth is requested. Trying to authenticate')
            mdb.authenticate(user, password, source=authdb)
    except pymongo.errors.PyMongoError:
        log.error('Error connecting to database %s', database)
        return False

    return conn


def _conn_is_alive(conn, dbname):
    try:
        if isinstance(conn, pymongo.MongoClient):
            return conn[dbname].command('ping')['ok'] == 1
    except pymongo.errors.PyMongoError:
        pass

    return False


def _to_dict(objects):
    '''
    Potentially interprets a string as JSON for usage with mongo
    '''
    try:
        if isinstance(objects, six.string_types):
            objects = json.loads(objects)
    except ValueError as err:
        log.error("Could not parse objects: %s", err)
        raise err

    return objects


def is_primary(user=None, password=None, host=None, port=None, authdb=None):
    """
    Check if given mongodb instance is primary
    """
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        return False

    try:
        res = conn.is_primary
        if res:
            log.info('%s is primary', host)
        return res
    except pymongo.errors.PyMongoError as err:
        log.error(err)


def find_master(hosts, user=None, password=None, port=27018, timeout=60):
    """
    Find master within hosts
    (if hosts is not list we assume it is pillar path)
    """
    if isinstance(hosts, list):
        host_list = hosts
    else:
        host_list = __salt__['pillar.get'](hosts)

    if not host_list:
        return

    deadline = time.time() + timeout
    while time.time() < deadline:
        for host in host_list:
            if is_primary(user=user, password=password, host=host, port=port):
                return host
        time.sleep(1)


def find_rs_primary(secondary_catch_up_period_secs=None, timeout=None,
                    user=None, password=None, host=None, port=None,
                    authdb='admin'):
    """
    Find primary within replicaset
    """

    if timeout is None:
        timeout = DEFAULT_TIMEOUT_SECS
    timeout_ms = timeout * 1000

    if secondary_catch_up_period_secs is None:
        secondary_catch_up_period_secs = SECONDARY_CATCH_UP_PERIOD_SECS

    options = dict((opt, timeout_ms) for opt in ('connectTimeoutMS',
                                                 'socketTimeoutMS',
                                                 'serverSelectionTimeoutMS'))
    conn = _connect(user, password, host, port, options=options)
    if not conn:
        return None

    try:
        server_selection_timeout = conn.server_selection_timeout
        heartbeat_timeout_secs = conn.admin.command({'replSetGetConfig': 1})[
            'config']['settings']['heartbeatTimeoutSecs']
    except pymongo.errors.PyMongoError as exc:
        log.error('Can not get replSetGetConfig: %s', exc, exc_info=True)
        return None

    # https://docs.mongodb.com/manual/reference/method/rs.stepDown/#behavior
    deadline = time.time() \
        + secondary_catch_up_period_secs \
        + server_selection_timeout \
        + heartbeat_timeout_secs

    while time.time() < deadline:
        try:
            if not _conn_is_alive(conn, 'admin'):
                conn = _connect(user, password, host, port, authdb,
                                options=options)
            replset_members = conn.admin.command('replSetGetStatus')[
                'members']
            primaries = [node for node in replset_members
                         if node['stateStr'] == 'PRIMARY']
            if len(primaries) == 1:
                primary = primaries.pop()
                return primary['name'].split(':')[0]
            elif len(primaries) > 1:
                log.critical('Several primaries were found: %s',
                             ', '.join(primaries))

            log.debug('Waiting for primary')
        except Exception as err:
            log.warning('Failed while waiting for primary with error: %s',
                        err, exc_info=True)

        time.sleep(timeout)


def get_rs_members(user=None, password=None, host=None, port=None, authdb=None):
    """
    Get list of replicaSet members
    """
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        return []

    try:
        status = conn.admin.command('replSetGetStatus')
        return [x['name'].split(':')[0] for x in status['members']]
    except pymongo.errors.PyMongoError as err:
        log.error(err)
        return []


def db_list(user=None, password=None, host=None, port=None, authdb=None):
    '''
    List all Mongodb databases

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.db_list <user> <password> <host> <port>
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        log.info('Listing databases')
        return conn.database_names()
    except pymongo.errors.PyMongoError as err:
        log.error(err)
        return six.text_type(err)


def db_exists(name, user=None, password=None, host=None, port=None, authdb=None):
    '''
    Checks if a database exists in Mongodb

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.db_exists <name> <user> <password> <host> <port>
    '''
    dbs = db_list(user, password, host, port, authdb=authdb)

    if isinstance(dbs, six.string_types):
        return False

    return name in dbs


def db_remove(name, user=None, password=None, host=None, port=None, authdb=None):
    '''
    Remove a Mongodb database

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.db_remove <name> <user> <password> <host> <port>
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        log.info('Removing database %s', name)
        conn.drop_database(name)
    except pymongo.errors.PyMongoError as err:
        log.error('Removing database %s failed with error: %s', name, err)
        return six.text_type(err)

    return True


def _version(mdb):
    return mdb.command('buildInfo')['version']


def version(user=None, password=None, host=None, port=None, database='admin', authdb=None):
    '''
    Get MongoDB instance version

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.version <user> <password> <host> <port> <database>
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        err_msg = "Failed to connect to MongoDB database {0}:{1}".format(host, port)
        log.error(err_msg)
        return (False, err_msg)

    try:
        mdb = pymongo.database.Database(conn, database)
        return _version(mdb)
    except pymongo.errors.PyMongoError as err:
        log.error('Listing users failed with error: %s', err)
        return six.text_type(err)


def user_find(name, user=None, password=None, host=None, port=None,
                database='admin', authdb=None):
    '''
    Get single user from MongoDB

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.user_find <name> <user> <password> <host> <port> <database> <authdb>
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        err_msg = "Failed to connect to MongoDB database {0}:{1}".format(host, port)
        log.error(err_msg)
        return (False, err_msg)

    mdb = pymongo.database.Database(conn, database)
    try:
        return mdb.command("usersInfo", name)["users"]
    except pymongo.errors.PyMongoError as err:
        log.error('Listing users failed with error: %s', err)
        return (False, six.text_type(err))


def user_auth(user=None, password=None, host=None, port=None, authdb=None):
    '''
    Get single user from MongoDB

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.user_auth <user> <password> <host> <port> <authdb>
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        err_msg = "Failed to connect to MongoDB database {0}:{1}".format(host, port)
        log.error(err_msg)
        return (False, err_msg)

    mdb = pymongo.database.Database(conn, authdb)
    try:
        result =  (True, mdb.command("usersInfo", user)["users"])
        conn.close()
        return result
    except pymongo.errors.PyMongoError as err:
        log.error('Listing users failed with error: %s', err)
        return (False, six.text_type(err))


def user_list(user=None, password=None, host=None, port=None, database='admin', authdb=None):
    '''
    List users of a Mongodb database

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.user_list <user> <password> <host> <port> <database>
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        log.info('Listing users')
        mdb = pymongo.database.Database(conn, database)

        output = []

        #> dbaas
        #> if _LooseVersion(mongodb_version) >= _LooseVersion('2.6'):
        if True:
            for user in mdb.command('usersInfo')['users']:
                output.append(
                    {'user': user['user'],
                     'roles': user['roles']}
                )
        else:
            for user in mdb.system.users.find():
                output.append(
                    {'user': user['user'],
                     'readOnly': user.get('readOnly', 'None')}
                )
        return output

    except pymongo.errors.PyMongoError as err:
        log.error('Listing users failed with error: %s', err)
        return six.text_type(err)


def user_exists(name, user=None, password=None, host=None, port=None,
                database='admin', authdb=None):
    '''
    Checks if a user exists in Mongodb

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.user_exists <name> <user> <password> <host> <port> <database>
    '''
    users = user_list(user, password, host, port, database, authdb)

    if isinstance(users, six.string_types):
        return 'Failed to connect to mongo database'

    for user in users:
        if name == dict(user).get('user'):
            return True

    return False


def user_create(name, passwd, user=None, password=None, host=None, port=None,
                database='admin', authdb=None, roles=None):
    '''
    Create a Mongodb user

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.user_create <user_name> <user_password> <roles> <user> <password> <host> <port> <database>
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    log.debug('%s', conn.admin.command('replSetGetStatus'))
    if not roles:
        roles = []

    try:
        log.info('Creating user %s', name)
        mdb = pymongo.database.Database(conn, database)
        mdb.add_user(name, passwd, roles=roles)
    except pymongo.errors.PyMongoError as err:
        log.error('Creating user %s failed with error: %s', name, err)
        return six.text_type(err)
    return True


def user_remove(name, user=None, password=None, host=None, port=None,
                database='admin', authdb=None):
    '''
    Remove a Mongodb user

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.user_remove <name> <user> <password> <host> <port> <database>
    '''
    conn = _connect(user, password, host, port)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        log.info('Removing user %s', name)
        mdb = pymongo.database.Database(conn, database)
        mdb.remove_user(name)
    except pymongo.errors.PyMongoError as err:
        log.error('Creating database %s failed with error: %s', name, err)
        return six.text_type(err)

    return True


def user_roles_exists(name, roles, database, user=None, password=None, host=None,
                      port=None, authdb=None):
    '''
    Checks if a user of a Mongodb database has specified roles

    CLI Examples:

    .. code-block:: bash

        salt '*' mongodb.user_roles_exists johndoe '["readWrite"]' dbname admin adminpwd localhost 27017

    .. code-block:: bash

        salt '*' mongodb.user_roles_exists johndoe '[{"role": "readWrite", "db": "dbname" }, {"role": "read", "db": "otherdb"}]' dbname admin adminpwd localhost 27017
    '''
    try:
        roles = _to_dict(roles)
    except Exception:
        return 'Roles provided in wrong format'

    users = user_list(user, password, host, port, database, authdb)

    if isinstance(users, six.string_types):
        return 'Failed to connect to mongo database'

    for user in users:
        if name == dict(user).get('user'):
            for role in roles:
                # if the role was provided in the shortened form, we convert it to a long form
                if not isinstance(role, dict):
                    role = {'role': role, 'db': database}
                if role not in dict(user).get('roles', []):
                    return False
            return True

    return False


def user_grant_roles(name, roles, database, user=None, password=None, host=None,
                     port=None, authdb=None):
    '''
    Grant one or many roles to a Mongodb user

    CLI Examples:

    .. code-block:: bash

        salt '*' mongodb.user_grant_roles johndoe '["readWrite"]' dbname admin adminpwd localhost 27017

    .. code-block:: bash

        salt '*' mongodb.user_grant_roles janedoe '[{"role": "readWrite", "db": "dbname" }, {"role": "read", "db": "otherdb"}]' dbname admin adminpwd localhost 27017
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        roles = _to_dict(roles)
    except Exception:
        return 'Roles provided in wrong format'

    try:
        log.info('Granting roles %s to user %s', roles, name)
        mdb = pymongo.database.Database(conn, database)
        mdb.command("grantRolesToUser", name, roles=roles)
    except pymongo.errors.PyMongoError as err:
        log.error('Granting roles %s to user %s failed with error: %s', roles, name, err)
        return six.text_type(err)

    return True


def user_revoke_roles(name, roles, database, user=None, password=None, host=None,
                      port=None, authdb=None):
    '''
    Revoke one or many roles to a Mongodb user

    CLI Examples:

    .. code-block:: bash

        salt '*' mongodb.user_revoke_roles johndoe '["readWrite"]' dbname admin adminpwd localhost 27017

    .. code-block:: bash

        salt '*' mongodb.user_revoke_roles janedoe '[{"role": "readWrite", "db": "dbname" }, {"role": "read", "db": "otherdb"}]' dbname admin adminpwd localhost 27017
    '''
    conn = _connect(user, password, host, port, authdb=authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        roles = _to_dict(roles)
    except Exception:
        return 'Roles provided in wrong format'

    try:
        log.info('Revoking roles %s from user %s', roles, name)
        mdb = pymongo.database.Database(conn, database)
        mdb.command("revokeRolesFromUser", name, roles=roles)
    except pymongo.errors.PyMongoError as err:
        log.error('Revoking roles %s from user %s failed with error: %s', roles, name, err)
        return six.text_type(err)

    return True


def insert(objects, collection, user=None, password=None,
           host=None, port=None, database='admin', authdb=None):
    '''
    Insert an object or list of objects into a collection

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.insert '[{"foo": "FOO", "bar": "BAR"}, {"foo": "BAZ", "bar": "BAM"}]' mycollection <user> <password> <host> <port> <database>

    '''
    conn = _connect(user, password, host, port, database, authdb)
    if not conn:
        return "Failed to connect to mongo database"

    try:
        objects = _to_dict(objects)
    except Exception as err:
        return err

    try:
        log.info("Inserting %r into %s.%s", objects, database, collection)
        mdb = pymongo.database.Database(conn, database)
        col = getattr(mdb, collection)
        ids = col.insert(objects)
        return ids
    except pymongo.errors.PyMongoError as err:
        log.error("Inserting objects %r failed with error %s", objects, err)
        return err


def update_one(objects, collection, user=None, password=None, host=None, port=None, database='admin', authdb=None):
    '''
    Update an object into a collection
    http://api.mongodb.com/python/current/api/pymongo/collection.html#pymongo.collection.Collection.update_one

    .. versionadded:: 2016.11.0

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.update_one '{"_id": "my_minion"} {"bar": "BAR"}' mycollection <user> <password> <host> <port> <database>

    '''
    conn = _connect(user, password, host, port, database, authdb)
    if not conn:
        return "Failed to connect to mongo database"

    objects = six.text_type(objects)
    objs = re.split(r'}\s+{', objects)

    if len(objs) is not 2:
        return "Your request does not contain a valid " + \
            "'{_\"id\": \"my_id\"} {\"my_doc\": \"my_val\"}'"

    objs[0] = objs[0] + '}'
    objs[1] = '{' + objs[1]

    document = []

    for obj in objs:
        try:
            obj = _to_dict(obj)
            document.append(obj)
        except Exception as err:
            return err

    _id_field = document[0]
    _update_doc = document[1]

    # need a string to perform the test, so using objs[0]
    test_f = find(collection,
                  objs[0],
                  user,
                  password,
                  host,
                  port,
                  database,
                  authdb)
    if not isinstance(test_f, list):
        return 'The find result is not well formatted. An error appears; cannot update.'
    elif len(test_f) < 1:
        return 'Did not find any result. You should try an insert before.'
    elif len(test_f) > 1:
        return 'Too many results. Please try to be more specific.'
    else:
        try:
            log.info("Updating %r into %s.%s", _id_field, database, collection)
            mdb = pymongo.database.Database(conn, database)
            col = getattr(mdb, collection)
            ids = col.update_one(_id_field, {'$set': _update_doc})
            nb_mod = ids.modified_count
            return "{0} objects updated".format(nb_mod)
        except pymongo.errors.PyMongoError as err:
            log.error('Updating object %s failed with error %s', objects, err)
            return err


def find(collection, query=None, user=None, password=None,
         host=None, port=None, database='admin', authdb=None):
    '''
    Find an object or list of objects in a collection

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.find mycollection '[{"foo": "FOO", "bar": "BAR"}]' <user> <password> <host> <port> <database>

    '''
    conn = _connect(user, password, host, port, database, authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        query = _to_dict(query)
    except Exception as err:
        return err

    try:
        log.info("Searching for %r in %s", query, collection)
        mdb = pymongo.database.Database(conn, database)
        col = getattr(mdb, collection)
        ret = col.find(query)
        return list(ret)
    except pymongo.errors.PyMongoError as err:
        log.error("Searching objects failed with error: %s", err)
        return err


def remove(collection, query=None, user=None, password=None,
           host=None, port=None, database='admin', w=1, authdb=None):
    '''
    Remove an object or list of objects into a collection

    CLI Example:

    .. code-block:: bash

        salt '*' mongodb.remove mycollection '[{"foo": "FOO", "bar": "BAR"}, {"foo": "BAZ", "bar": "BAM"}]' <user> <password> <host> <port> <database>

    '''
    conn = _connect(user, password, host, port, database, authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        query = _to_dict(query)
    except Exception as err:
        return _get_error_message(err)

    try:
        log.info("Removing %r from %s", query, collection)
        mdb = pymongo.database.Database(conn, database)
        col = getattr(mdb, collection)
        ret = col.remove(query, w=w)
        return "{0} objects removed".format(ret['n'])
    except pymongo.errors.PyMongoError as err:
        log.error("Removing objects failed with error: %s", _get_error_message(err))
        return _get_error_message(err)


def replset_add(hostport=None, arbiter=None, force=None, user=None, password=None, host=None, port=None, authdb=None):
    conn = _connect(user, password, host, port, authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    if conn.local.system.replset.count() > 1:
        return 'error: local.system.replset has unexpected contents'

    rs_conf = conn.local.system.replset.find_one()
    if not rs_conf:
        return 'No config object retrievable from local.system.replset'

    log.debug('replset config: %s', rs_conf)
    rs_conf['version'] += 1

    max_id = 0
    for rs_member in rs_conf['members']:
        if rs_member['_id'] > max_id:
            max_id = rs_member['_id']

    if not isinstance(hostport, str):
        return 'Expected a host-and-port string, but got {0}'.format(json.loads(hostport))    

    new_member_cfg = {'_id': max_id + 1, 'host': hostport}
    if arbiter is True:
        new_member_cfg['arbiteriterOnly'] = True

    if '_id' not in new_member_cfg:
        new_member_cfg['_id'] = max_id + 1

    try:
        rs_conf['members'].append(new_member_cfg)
        ret = conn.admin.command('replSetReconfig', rs_conf, force=force)
        log.debug('Node added: %s', ret)
        time.sleep(RS_CHANGE_WAIT_SECS)
        return ret['ok'] == 1 or ret
    except pymongo.errors.PyMongoError as err:
        log.error('Replicaset config update failed with error: %s', _get_error_message(err))
        return _get_error_message(err)


def replset_initiated(user=None, password=None, host=None, port=None, authdb=None):
    conn = _connect(user, password, host, port, authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        return bool(conn.admin.command('replSetGetStatus'))
    except pymongo.errors.PyMongoError as err:
        if 'no replset config has been received' in str(err):
            return False
        else:
            log.error('Replicaset status fetch failed with error: %s', _get_error_message(err))
            return _get_error_message(err)


def replset_initiate(user=None, password=None, host=None, port=None, authdb=None):
    conn = _connect(user, password, host, port, authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    try:
        rs_conf = conn.admin.command('replSetGetStatus')
    except pymongo.errors.PyMongoError as err:
        if 'no replset config has been received' not in str(err):
            return _get_error_message(err)

    try:
        ret = conn.admin.command('replSetInitiate', {})
        log.debug('Replicaset initiated: %s', ret)
        time.sleep(RS_CHANGE_WAIT_SECS)
        return ret['ok'] == 1 or ret
    except pymongo.errors.PyMongoError as err:
        log.error('Replicaset config update failed with error: %s', _get_error_message(err))
        return _get_error_message(err)


def replset_remove(hostport, force=None, user=None, password=None, host=None, port=None, authdb=None):
    conn = _connect(user, password, host, port, authdb)
    if not conn:
        return 'Failed to connect to mongo database'

    if conn.local.system.replset.count() > 1:
        return 'error: local.system.replset has unexpected contents'

    rs_conf = conn.local.system.replset.find_one()
    if not rs_conf:
        return 'No config object retrievable from local.system.replset'

    log.debug('replset config: %s', rs_conf)
    rs_conf['version'] += 1

    if not isinstance(hostport, str):
        return 'Expected a host-and-port string of arbiter, but got {0}'.format(json.loads(hostport))

    try:
        for i, rs_member in enumerate(rs_conf['members']):
            log.debug('%s', rs_member)
            if rs_member['host'] == hostport:
                rs_conf['members'].pop(i)
                break

        log.debug('Applying rs_conf: %s', rs_conf)
        ret = conn.admin.command('replSetReconfig', rs_conf, force=force)
        log.debug('rs_conf apply result: %s', ret)
        return ret['ok'] == 1 or ret
    except pymongo.errors.PyMongoError as err:
        log.error('Replicaset config update failed with error: %s', _get_error_message(err))
        return _get_error_message(err)


def is_alive(tries=None, timeout=None, user=None, password=None, host=None, port=None, authdb=None):
    if tries is None:
        tries = 1

    if timeout is None:
        timeout = 5
    timeout_ms = timeout * 1000

    if authdb is None:
        authdb = 'admin'

    options = dict((opt, timeout_ms) for opt in ('connectTimeoutMS', 'socketTimeoutMS', 'serverSelectionTimeoutMS'))
    for _ in range(tries):
        conn = _connect(user, password, host, port, authdb, options=options)
        ts = time.time()
        try:
            return conn[authdb].command('ping')['ok'] == 1
        except pymongo.errors.PyMongoError as err:
            log.warning('ping command failed with error: %s', _get_error_message(err))
            sleep_seconds = timeout - (time.time() - ts)
            if sleep_seconds > 0:
                time.sleep(sleep_seconds)
    return False


def rs_step_down(step_down_secs=None, secondary_catch_up_period_secs=None,
                 timeout=None, user=None, password=None, host=None, port=None,
                 authdb='admin'):
    if timeout is None:
        timeout = DEFAULT_TIMEOUT_SECS
    timeout_ms = timeout * 1000

    if step_down_secs is None:
        step_down_secs = 60
    if secondary_catch_up_period_secs is None:
        secondary_catch_up_period_secs = SECONDARY_CATCH_UP_PERIOD_SECS

    options = dict((opt, timeout_ms) for opt in ('connectTimeoutMS',
                                                 'socketTimeoutMS',
                                                 'serverSelectionTimeoutMS'))
    conn = _connect(user, password, host, port, authdb, options=options)
    if not conn:
        return 'Failed to connect to mongo database'

    if not conn.is_primary:
        return True

    try:
        log.debug('Executing stepdown')
        stepdown = conn.admin.command('replSetStepDown',
                                      step_down_secs,
                                      secondary_catch_up_period_secs)
        if stepdown['ok'] != 1:
            log.error('Can not stepdown: check rs members state')
            return False

    except pymongo.errors.AutoReconnect:
        # https://docs.mongodb.com/manual/reference/method/rs.stepDown/#behavior
        primary = find_rs_primary(secondary_catch_up_period_secs, timeout,
                                  user, password, host, port, authdb)
        log.debug('New PRIMARY is %s', primary)
        return host != primary

    except pymongo.errors.OperationFailure as err:
        log.critical('Failed stepping down with error: %s', str(err))

    return False


def get_rs_state_str(timeout=None, user=None, password=None, host=None,
                     port=None, authdb='admin'):
    if timeout is None:
        timeout = 5
    timeout_ms = timeout * 1000

    options = dict((opt, timeout_ms) for opt in ('connectTimeoutMS',
                                                 'socketTimeoutMS',
                                                 'serverSelectionTimeoutMS'))

    try:
        conn = _connect(user, password, host, port, authdb,
                        options=options)
        if not conn:
            return None

        my_state = conn.admin.command('replSetGetStatus')['myState']
        return MEMBER_INT_STATES[my_state]
    except Exception as err:
        log.warning('Failed to get role with error: %s', err, exc_info=True)


def wait_for_rs_role(state_str, deadline=None, timeout=None, user=None,
                     password=None, host=None, port=None, authdb='admin'):
    if timeout is None:
        timeout = 5
    timeout_ms = timeout * 1000

    if deadline is None:
        deadline = 10

    options = dict((opt, timeout_ms) for opt in ('connectTimeoutMS',
                                                 'socketTimeoutMS',
                                                 'serverSelectionTimeoutMS'))
    conn = None
    deadline += int(time.time())
    while time.time() < deadline:
        try:
            if not _conn_is_alive(conn, authdb):
                conn = _connect(user, password, host, port, authdb,
                                options=options)

            my_state = conn.admin.command('replSetGetStatus')['myState']
            if MEMBER_STR_STATES[state_str] == my_state:
                return True
            log.debug('Waiting for become: %s', state_str)

        except Exception as err:
            log.warning('Failed while waiting for role with error:'
                        ' %s', err, exc_info=True)

        time.sleep(5)
    return False


def wait_for_rs_joined(deadline=None, timeout=None, user=None,
                       password=None, host=None, port=None, authdb='admin'):
    if timeout is None:
        timeout = 5
    timeout_ms = timeout * 1000

    if deadline is None:
        deadline = 10

    options = dict((opt, timeout_ms) for opt in ('connectTimeoutMS',
                                                 'socketTimeoutMS',
                                                 'serverSelectionTimeoutMS'))
    conn = None
    deadline += int(time.time())
    while time.time() < deadline:
        try:
            if not _conn_is_alive(conn, authdb):
                conn = _connect(user, password, host, port, authdb,
                                options=options)

            my_state = conn.admin.command('replSetGetStatus')['myState']
            if my_state not in MEMBER_ERROR_STATES.values():
                return True
            log.debug('Waiting for normal state')

        except Exception as err:
            log.warning('Failed while waiting for role with error:'
                        ' %s', err, exc_info=True)

        time.sleep(5)
    return False
