# -*- coding: utf-8 -*-
'''
Management of Mongodb users and databases
=========================================

.. note::
    This module requires PyMongo to be installed.
'''

# Import Python libs
from __future__ import absolute_import, print_function, unicode_literals

# Define the module's virtual name
__virtualname__ = 'mongodb'


def __virtual__():
    if 'mongodb.user_exists' not in __salt__:
        return False
    return __virtualname__


def database_absent(name,
           user=None,
           password=None,
           host=None,
           port=None,
           authdb=None):
    '''
    Ensure that the named database is absent. Note that creation doesn't make sense in MongoDB.

    name
        The name of the database to remove

    user
        The user to connect as (must be able to create the user)

    password
        The password of the user

    host
        The host to connect to

    port
        The port to connect to

    authdb
        The database in which to authenticate
    '''
    ret = {'name': name,
           'changes': {},
           'result': True,
           'comment': ''}

    #check if database exists and remove it
    if __salt__['mongodb.db_exists'](name, user, password, host, port, authdb=authdb):
        if __opts__['test']:
            ret['result'] = None
            ret['comment'] = ('Database {0} is present and needs to be removed'
                    ).format(name)
            return ret
        if __salt__['mongodb.db_remove'](name, user, password, host, port, authdb=authdb):
            ret['comment'] = 'Database {0} has been removed'.format(name)
            ret['changes'][name] = 'Absent'
            return ret

    # fallback
    ret['comment'] = ('User {0} is not present, so it cannot be removed'
            ).format(name)
    return ret


def user_present(name,
            passwd,
            database="admin",
            user=None,
            password=None,
            host="localhost",
            port=27017,
            authdb=None):
    '''
    Ensure that the user is present with the specified properties

    name
        The name of the user to manage

    passwd
        The password of the user to manage

    user
        MongoDB user with sufficient privilege to create the user

    password
        Password for the admin user specified with the ``user`` parameter

    host
        The hostname/IP address of the MongoDB server

    port
        The port on which MongoDB is listening

    database
        The database in which to create the user

        .. note::
            If the database doesn't exist, it will be created.

    authdb
        The database in which to authenticate

    Example:

    .. code-block:: yaml

        mongouser-myapp:
          mongodb.user_present:
          - name: myapp
          - passwd: password-of-myapp
          # Connect as admin:sekrit
          - user: admin
          - password: sekrit

    '''
    ret = {'name': name,
           'changes': {},
           'result': True,
           'comment': 'User {0} is already present'.format(name)}

    # Check for valid port
    try:
        port = int(port)
    except TypeError:
        ret['result'] = False
        ret['comment'] = 'Port ({0}) is not an integer.'.format(port)
        return ret

    # check if user exists
    # fixed 'user_exists' -> 'user_auth'
    user_exists, user_roles = __salt__['mongodb.user_auth'](name, passwd, host, port, database)
    if user_exists is True:
        return ret

    # if the check does not return a boolean, return an error
    # this may be the case if there is a database connection error
    if not isinstance(user_exists, bool):
        ret['comment'] = user_exists
        ret['result'] = False
        return ret

    if __opts__['test']:
        ret['result'] = None
        ret['comment'] = ('User {0} is not present and needs to be created'
                ).format(name)
        return ret
    # The user is not present, make it!
    if __salt__['mongodb.user_create'](name, passwd, user, password, host, port, database=database, authdb=authdb):
        ret['comment'] = 'User {0} has been created'.format(name)
        ret['changes'][name] = 'Present'
    else:
        ret['comment'] = 'Failed to create database {0}'.format(name)
        ret['result'] = False

    return ret


def user_create(name,
            passwd,
            database="admin",
            user=None,
            password=None,
            host="localhost",
            port=27017,
            authdb=None,
            roles=None):
    '''
    Ensure that the user is present with the specified properties

    name
        The name of the user to manage

    passwd
        The password of the user to manage

    user
        MongoDB user with sufficient privilege to create the user

    password
        Password for the admin user specified with the ``user`` parameter

    host
        The hostname/IP address of the MongoDB server

    port
        The port on which MongoDB is listening

    database
        The database in which to create the user

        .. note::
            If the database doesn't exist, it will be created.

    authdb
        The database in which to authenticate



    Example:

    .. code-block:: yaml

        mongouser-myapp:
          mongodb.user_present:
          - name: myapp
          - passwd: password-of-myapp
          # Connect as admin:sekrit
          - user: admin
          - password: sekrit

    '''
    ret = {'name': name,
           'changes': {},
           'result': True,
           'comment': 'User {0} is already present'.format(name)}

    # Check for valid port
    try:
        port = int(port)
    except TypeError:
        ret['result'] = False
        ret['comment'] = 'Port ({0}) is not an integer.'.format(port)
        return ret

    if __opts__['test']:
        ret['result'] = None
        return ret

    # The user is not present, make it!
    if __salt__['mongodb.user_create'](name, passwd, user, password, host,
                                       port, database=database, authdb=authdb,
                                       roles=roles):
        ret['comment'] = 'User {0} has been created'.format(name)
        ret['changes'][name] = 'Present'
    else:
        ret['comment'] = 'Failed to create user {0}'.format(name)
        ret['result'] = False

    return ret


def user_absent(name,
           user=None,
           password=None,
           host=None,
           port=None,
           database="admin",
           authdb=None):
    '''
    Ensure that the named user is absent

    name
        The name of the user to remove

    user
        MongoDB user with sufficient privilege to create the user

    password
        Password for the admin user specified by the ``user`` parameter

    host
        The hostname/IP address of the MongoDB server

    port
        The port on which MongoDB is listening

    database
        The database from which to remove the user specified by the ``name``
        parameter

    authdb
        The database in which to authenticate
    '''
    ret = {'name': name,
           'changes': {},
           'result': True,
           'comment': ''}

    #check if user exists and remove it
    user_exists = __salt__['mongodb.user_exists'](name, user, password, host, port, database=database, authdb=authdb)
    if user_exists is True:
        if __opts__['test']:
            ret['result'] = None
            ret['comment'] = ('User {0} is present and needs to be removed'
                    ).format(name)
            return ret
        if __salt__['mongodb.user_remove'](name, user, password, host, port, database=database, authdb=authdb):
            ret['comment'] = 'User {0} has been removed'.format(name)
            ret['changes'][name] = 'Absent'
            return ret

    # if the check does not return a boolean, return an error
    # this may be the case if there is a database connection error
    if not isinstance(user_exists, bool):
        ret['comment'] = user_exists
        ret['result'] = False
        return ret

    # fallback
    ret['comment'] = ('User {0} is not present, so it cannot be removed'
            ).format(name)
    return ret


def _roles_to_set(roles, database):
    ret = set()
    for r in roles:
        if isinstance(r, dict):
            if r['db'] == database:
                ret.add(r['role'])
        else:
            ret.add(r)
    return ret


def _user_roles_to_set(user_list, name, database):
    ret = set()

    for item in user_list:
        if item['user'] == name:
            ret = ret.union(_roles_to_set(item['roles'], database))
    return ret


def user_grant_roles(name, roles,
            database="admin",
            user=None,
            password=None,
            host="localhost",
            port=27017,
            authdb=None):

    '''
    Ensure that the named user is granted certain roles

    name
        The name of the user to remove

    roles
        The roles to grant to the user

    user
        MongoDB user with sufficient privilege to create the user

    password
        Password for the admin user specified by the ``user`` parameter

    host
        The hostname/IP address of the MongoDB server

    port
        The port on which MongoDB is listening

    database
        The database from which to remove the user specified by the ``name``
        parameter

    authdb
        The database in which to authenticate
    '''

    ret = {'name': name,
           'changes': {},
           'result': False,
           'comment': ''}

    if not isinstance(roles, (list, tuple)):
        roles = [roles]

    if not roles:
        ret['result'] = True
        ret['comment'] = "nothing to do (no roles given)"
        return ret

    # Check for valid port
    try:
        port = int(port)
    except TypeError:
        ret['result'] = False
        ret['comment'] = 'Port ({0}) is not an integer.'.format(port)
        return ret

    # check if grant exists
    user_roles_exists = __salt__['mongodb.user_roles_exists'](name, roles, database,
        user=user, password=password, host=host, port=port, authdb=authdb)
    if user_roles_exists is True:
        ret['result'] = True
        ret['comment'] = "Roles already assigned"
        return ret

    user_list = __salt__['mongodb.user_list'](database=database,
        user=user, password=password, host=host, port=port, authdb=authdb)

    user_set = _user_roles_to_set(user_list, name, database)
    roles_set = _roles_to_set(roles, database)
    diff = roles_set - user_set

    if __opts__['test']:
        ret['result'] = None
        ret['comment'] = "Would have modified roles (missing: {0})".format(diff)
        return ret

    # The user is not present, make it!
    if __salt__['mongodb.user_grant_roles'](name, roles, database,
        user=user, password=password, host=host, port=port, authdb=authdb):
        ret['comment'] = 'Granted roles to {0} on {1}'.format(name, database)
        ret['changes'][name] = ['{0} granted'.format(i) for i in diff]
        ret['result'] = True
    else:
        ret['comment'] = 'Failed to grant roles ({2}) to {0} on {1}'.format(name, database, diff)

    return ret


def user_set_roles(name, roles,
            database="admin",
            user=None,
            password=None,
            host="localhost",
            port=27017,
            authdb=None):

    '''
    Ensure that the named user has the given roles and no other roles

    name
        The name of the user to remove

    roles
        The roles the given user should have

    user
        MongoDB user with sufficient privilege to create the user

    password
        Password for the admin user specified by the ``user`` parameter

    host
        The hostname/IP address of the MongoDB server

    port
        The port on which MongoDB is listening

    database
        The database from which to remove the user specified by the ``name``
        parameter

    authdb
        The database in which to authenticate
    '''

    ret = {'name': name,
           'changes': {},
           'result': False,
           'comment': ''}

    if not isinstance(roles, (list, tuple)):
        roles = [roles]

    if not roles:
        ret['result'] = True
        ret['comment'] = "nothing to do (no roles given)"
        return ret

    # Check for valid port
    try:
        port = int(port)
    except TypeError:
        ret['result'] = False
        ret['comment'] = 'Port ({0}) is not an integer.'.format(port)
        return ret

    user_list = __salt__['mongodb.user_list'](database=database,
        user=user, password=password, host=host, port=port, authdb=authdb)

    user_set = _user_roles_to_set(user_list, name, database)
    roles_set = _roles_to_set(roles, database)
    to_grant = list(roles_set - user_set)
    to_revoke = list(user_set - roles_set)

    if not to_grant and not to_revoke:
        ret['result'] = True
        ret['comment'] = "User {0} has the appropriate roles on {1}".format(name, database)
        return ret

    if __opts__['test']:
        lsg = ', '.join(to_grant)
        lsr = ', '.join(to_revoke)
        ret['result'] = None
        ret['comment'] = "Would have modified roles (grant: {0}; revoke: {1})".format(lsg, lsr)
        return ret

    ret['changes'][name] = changes = {}

    if to_grant:
        if not __salt__['mongodb.user_grant_roles'](name, to_grant, database,
            user=user, password=password, host=host, port=port, authdb=authdb):
            ret['comment'] = "failed to grant some or all of {0} to {1} on {2}".format(to_grant, name, database)
            return ret
        else:
            changes['granted'] = list(to_grant)

    if to_revoke:
        if not __salt__['mongodb.user_revoke_roles'](name, to_revoke, database,
            user=user, password=password, host=host, port=port, authdb=authdb):
            ret['comment'] = "failed to revoke some or all of {0} to {1} on {2}".format(to_revoke, name, database)
            return ret
        else:
            changes['revoked'] = list(to_revoke)

    ret['result'] = True
    return ret


def replset_add(name,
           arbiter=None,
           force=None,
           user=None,
           password=None,
           host=None,
           port=None,
           authdb=None):
    '''
    Add member to replicaset.

    name
        The hostport to add

    arbiter
        Is new member is arbiter

    force
        Force replset update

    user
        The user to connect as (must be able to manage replset)

    password
        The password of the user

    host
        The host to connect to

    port
        The port to connect to

    authdb
        The database in which to authenticate
    '''
    ret = {'name': name,
           'changes': {},
           'result': False,
           'comment': ''}

    #  TODO: check rs is configured

    if __opts__['test']:
        ret['comment'] = 'Hostport needs to be added to replicaset: {name}'.format(name=name)
        ret['result'] = None
        return ret

    add_result =  __salt__['mongodb.replset_add'](name, arbiter, force, user, password, host, port, authdb)
    if add_result is True:
        ret['comment'] = 'Hostport {0} has been added to replicaset'.format(name)
        ret['result'] = True
        return ret

    ret['comment'] = add_result
    return ret


def replset_remove(name, force=None, user=None, password=None,
                   host=None, port=None, authdb=None):
    """
    Remove member from replicaset

    name
        The hostport to add

    user
        The user to connect as (must be able to manage replset)

    password
        The password of the user

    host
        The host to connect to

    port
        The port to connect to

    authdb
        The database in which to authenticate
    """
    ret = {'name': name,
           'changes': {},
           'result': False,
           'comment': ''}

    if __opts__['test']:
        ret['comment'] = 'Hostport needs to be removed from replicaset: {name}'.format(name=name)
        ret['result'] = None
        return ret

    if __salt__['mongodb.is_primary'](user, password, host, port, authdb):
        remove_result =  __salt__['mongodb.replset_remove'](name, force, user, password, host, port, authdb)
        ret['comment'] = 'Hostport {name} has been removed from replicaset'.format(name=name)
    else:
        remove_result = True
        ret['comment'] = 'Replica. Not changing replicaset configuration.'

    if remove_result is True:
        ret['result'] = True
        return ret

    ret['comment'] = remove_result
    return ret


def replset_initiate(name,
           user=None,
           password=None,
           port=None,
           authdb=None):
    '''
    Initiate replicaset

    name
        The host to initiate on

    user
        The user to connect as (must be able to manage replset)

    password
        The password of the user

    port
        The port to connect to

    authdb
        The database in which to authenticate
    '''
    ret = {'name': name,
           'changes': {},
           'result': False,
           'comment': ''}

    #  TODO: check rs is configured

    init_result =  __salt__['mongodb.replset_initiate'](user, password, name, port, authdb)
    if init_result is True:
        ret['comment'] = 'ReplicaSet has been initiated at {0}:{1}'.format(name, port)
        ret['result'] = True
        return ret

    ret['comment'] = init_result
    return ret


def alive(name,
          tries=None,
          timeout=None,
          user=None,
          password=None,
          port=None,
          authdb=None):
    '''
    Check mongodb alive

    name
        The host to check

    tries
        TODO

    timeout
        TODO

    user
        The user to connect as (must be able to manage replset)

    password
        The password of the user

    port
        The port to connect to

    authdb
        The database in which to authenticate
    '''
    ret = {'name': name,
           'changes': {},
           'result': False,
           'comment': ''}

    is_alive = __salt__['mongodb.is_alive'](tries, timeout, user, password, name, port, authdb)
    if not is_alive:
        ret['comment'] = 'MongoDB at {0}:{1} seems dead'.format(name, port)
    ret['result'] = is_alive
    return ret


def rs_role_wait(name,
            state_str=None,
            deadline=None,
            timeout=None,
            user=None,
            password=None,
            port=None,
            authdb=None):
    '''
    Check mongodb role

    name
        The host to check

    state_str
        Desired member state

    deadline
        Wait state_str till given time

    timeout
        pymongo timeouts

    user
        The user to connect as (must be able to manage replset)

    password
        The password of the user

    port
        The port to connect to

    authdb
        The database in which to authenticate
    '''
    ret = {'name': name,
           'changes': {},
           'comment': '',
           'result': False,
           }

    rs_state_str = __salt__['mongodb.get_rs_state_str'](
        timeout, user, password, name, port, authdb)

    if rs_state_str is None:
        ret['comment'] = 'Failed to get state'.format(
            rs_state_str, state_str)
        ret['result'] = False
        return ret

    if __opts__['test']:
        if rs_state_str == state_str:
            ret['comment'] = 'Node is {0}'.format(state_str)
            ret['result'] = True
        else:
            ret['comment'] = 'Node is {0}, desired state is {1}'.format(
                rs_state_str, state_str)
            ret['result'] = None

        return ret

    success = __salt__['mongodb.wait_for_rs_role'](
        state_str, deadline, timeout, user, password, name, port, authdb)

    if not success:
        ret['comment'] = 'Node is {0}, desired state is {1}'.format(
                rs_state_str, state_str)
    ret['result'] = success
    return ret


def wait_for_rs_joined(name,
                       deadline=None,
                       timeout=None,
                       user=None,
                       password=None,
                       port=None,
                       authdb=None):
    '''
    Wait for replicaset join

    name
        The host to check

    deadline
        Wait state_str till given time

    timeout
        pymongo timeouts

    user
        The user to connect as (must be able to manage replset)

    password
        The password of the user

    port
        The port to connect to

    authdb
        The database in which to authenticate
    '''
    ret = {'name': name,
           'changes': {},
           'comment': '',
           'result': False,
           }

    rs_state_str = __salt__['mongodb.get_rs_state_str'](
        timeout, user, password, name, port, authdb)

    if rs_state_str is None:
        ret['comment'] = 'Failed to get state'.format(
            rs_state_str, state_str)
        ret['result'] = False
        return ret

    if __opts__['test']:
        if rs_state_str == state_str:
            ret['comment'] = 'Node has already joined replicaset'.\
                format(state_str)
            ret['result'] = True
        else:
            ret['comment'] = 'Node is {0}, need to wait for join'.format(
                rs_state_str, state_str)
            ret['result'] = None

        return ret

    success = __salt__['mongodb.wait_for_rs_joined'](
        deadline, timeout, user, password, name, port, authdb)

    if not success:
        ret['comment'] = 'Node has not joined replicaset yet'
    ret['result'] = success
    return ret


def rs_step_down(name,
                step_down_secs=None,
                secondary_catch_up_period_secs=None,
                timeout=None,
                user=None,
                password=None,
                port=None,
                authdb=None):
    '''
    Perform rs.stepDown

    name
        The host to step down

    step_down_secs
        The number of seconds to step down the primary, during which time
        the stepdown member is ineligible for becoming primary

    secondary_catch_up_period_secs
        The number of seconds that the mongod will wait for an electable
        secondary to catch up to the primary

    timeout
        pymongo timeouts

    user
        The user to connect as (must be able to manage replset)

    password
        The password of the user

    port
        The port to connect to

    authdb
        The database in which to authenticate
    '''
    ret = {'name': name,
           'changes': {},
           'result': False,
           'comment': ''}

    primary = __salt__['mongodb.find_rs_primary'](
        secondary_catch_up_period_secs, timeout, user,
        password, name, port, authdb
    )

    if primary is None:
        ret['comment'] = 'Can not find PRIMARY of replicaset'
        ret['result'] = False
        return ret

    if primary != name:
        ret['comment'] = '%s is not PRIMARY, no need to step down'.format(name)
        ret['result'] = True
        return ret

    if __opts__['test']:
        ret['comment'] = 'MongoDB is PRIMARY, need to step down'
        ret['result'] = None
        return ret

    rs_step_down = __salt__['mongodb.rs_step_down'](
        step_down_secs, secondary_catch_up_period_secs, timeout, user,
        password, name, port, authdb
    )
    ret['result'] = rs_step_down

    ret['comment'] = 'MongoDB could not step down'.format(name, port)\
        if not rs_step_down else 'MongoDB has stepped down'
    return ret
