#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from salt.ext import six


def __virtual__():
    if 'postgres.psql_exec' in __salt__:
        return 'postgresql_cmd'

    return False


def mod_run_check(onlyif, unless, **kwargs):
    """
    Execute the onlyif and unless logic.
    Return a result dict if:
    * onlyif failed (onlyif != 0)
    * unless succeeded (unless == 0)
    else return True
    """
    # never use VT for onlyif/unless executions because this will lead
    # to quote problems
    cmd_kwargs = {
        'ignore_retcode': True,
        'python_shell': True,
        'runas': 'postgres',
    }
    if onlyif is not None:
        if isinstance(onlyif, six.string_types):
            composed = __salt__['postgres.psql_compose'](onlyif, **kwargs)
            cmd = __salt__['cmd.retcode'](composed, **cmd_kwargs)
            if cmd != 0:
                return {
                    'comment': 'onlyif condition is false',
                    'skip_watch': True,
                    'result': True,
                }
        elif isinstance(onlyif, list):
            for entry in onlyif:
                composed = __salt__['postgres.psql_compose'](entry, **kwargs)
                cmd = __salt__['cmd.retcode'](composed, **cmd_kwargs)
                if cmd != 0:
                    return {
                        'comment':
                            'onlyif condition is false: {0}'.format(entry),
                        'skip_watch':
                            True,
                        'result':
                            True,
                    }
        elif not isinstance(onlyif, six.string_types):
            if not onlyif:
                return {
                    'comment': 'onlyif condition is false',
                    'skip_watch': True,
                    'result': True,
                }

    if unless is not None:
        if isinstance(unless, six.string_types):
            composed = __salt__['postgres.psql_compose'](unless, **kwargs)
            cmd = __salt__['cmd.retcode'](composed, **cmd_kwargs)
            if cmd == 0:
                return {
                    'comment': 'unless condition is true',
                    'skip_watch': True,
                    'result': True,
                }
        elif isinstance(unless, list):
            cmd = []
            for entry in unless:
                composed = __salt__['postgres.psql_compose'](entry, **kwargs)
                cmd.append(__salt__['cmd.retcode'](composed, **cmd_kwargs))
            if all([c == 0 for c in cmd]):
                return {
                    'comment': 'unless condition is true',
                    'skip_watch': True,
                    'result': True,
                }
        elif not isinstance(unless, six.string_types):
            if unless:
                return {
                    'comment': 'unless condition is true',
                    'skip_watch': True,
                    'result': True,
                }

    # No reason to stop, return True
    return True


def psql_exec(name,
              user=None,
              host=None,
              port=None,
              maintenance_db=None,
              password=None,
              runas=None,
              unless=None,
              onlyif=None):
    ret = {
        'name': name,
        'result': None,
        'comment': '',
        'changes': {},
    }

    cret = mod_run_check(
        onlyif,
        unless,
        user=user,
        host=host,
        port=port,
        maintenance_db=maintenance_db,
        password=password)

    if __opts__['test'] and cret is True:
        ret['comment'] = ('PostgreSQL command "{cmd}" '
                          'would have been executed'.format(cmd=name))
        return ret

    if isinstance(cret, dict):
        ret.update(cret)
        return ret

    code, stdout, stderr = __salt__['postgres.psql_exec'](
        name,
        user=user,
        host=host,
        port=port,
        maintenance_db=maintenance_db,
        password=password,
        runas=runas)

    ret['result'] = (code == 0)

    ret['changes']['psql_exec'] = name

    ret['comment'] = 'stdout: {stdout}\nstderr: {stderr}'.format(
        stdout=stdout, stderr=stderr)

    return ret


def psql_file(name,
              user=None,
              host=None,
              port=None,
              maintenance_db=None,
              password=None,
              runas=None,
              unless=None,
              onlyif=None):
    ret = {
        'name': name,
        'result': None,
        'comment': '',
        'changes': {},
    }

    cret = mod_run_check(
        onlyif,
        unless,
        user=user,
        host=host,
        port=port,
        maintenance_db=maintenance_db,
        password=password)

    if __opts__['test'] and cret is True:
        ret['comment'] = ('PostgreSQL file "{file_name}" '
                          'would have been executed'.format(file_name=name))
        return ret

    if isinstance(cret, dict):
        ret.update(cret)
        return ret

    code, stdout, stderr = __salt__['postgres.psql_file'](
        name,
        user=user,
        host=host,
        port=port,
        maintenance_db=maintenance_db,
        password=password,
        runas=runas)

    ret['result'] = (code == 0)

    ret['changes']['psql_file'] = name

    ret['comment'] = 'stdout: {stdout}\nstderr: {stderr}'.format(
        stdout=stdout, stderr=stderr)

    return ret


def set_master(name, hosts='data:dbaas:shard_hosts',
               user=None, port=None, password=None, timeout=60):
    """
    Find master within hosts and set `pg-master` in pillar
    """
    ret = {
        'name': name,
        'result': None,
        'comment': '',
        'changes': {},
    }

    master = __salt__['postgres.find_master'](
        hosts, user, port, password, timeout)

    if master:
        ret['result'] = True
        ret['comment'] = 'Found master at {name}'.format(name=master)
        __pillar__['pg-master'] = master
    else:
        ret['result'] = False
        ret['comment'] = 'Unable to find master'

    return ret


def populate_recovery_conf(name, application_name=None,
                           use_replication_slots=True):
    """
    Create recovery.conf
    """
    ret = {
        'name': name,
        'result': None,
        'comment': '',
        'changes': {},
    }

    master = __salt__['pillar.get'](
        'pg-master',
        __salt__['pillar.get']('data:pgsync:stream_from'))

    if os.path.exists(name):
        with open(name) as inp:
            content = inp.readlines()
        for line in content:
            if line.startswith('primary_conninfo = ') and master in line \
                    and application_name in line:
                ret['result'] = True
                ret['comment'] = 'recovery.conf already present'
                return ret

    cmd_kwargs = {
        'ignore_retcode': True,
        'python_shell': True,
        'runas': 'postgres',
    }

    cmd = ' '.join([
        '/usr/local/yandex/populate_recovery_conf.py',
        '-p', name,
        '-s' if not use_replication_slots else '',
        master,
    ])

    if __opts__['test']:
        ret['comment'] = '{cmd} would be executed'.format(cmd=cmd)
    else:
        res = __salt__['cmd.retcode'](cmd, **cmd_kwargs)
        if res == 0:
            ret['result'] = True
            ret['changes'] = {name: 'created'}
        else:
            ret['result'] = False
            ret['comment'] = '{cmd} exited with {code}'.format(
                cmd=cmd, code=res)

    return ret


def create_replication_slot(name, user=None, port=None,
                            password=None, control_path=None):
    """
    Create replication slot on master
    """
    ret = {
        'name': name,
        'result': None,
        'comment': '',
        'changes': {},
    }

    if control_path and os.path.exists(control_path):
        ret['result'] = True
        ret['comment'] = '{path} exists. Skipping slot creation'.format(
            path=control_path)

        return ret

    master = __salt__['pillar.get'](
        'pg-master',
        __salt__['pillar.get']('data:pgsync:stream_from'))

    query_result = __salt__['postgres.psql_query'](
        query=('SELECT slot_name FROM pg_replication_slots '
               "WHERE slot_name = '{name}'").format(name=name),
        host=master,
        user=user,
        port=port,
        password=password,
        maintenance_db='postgres')

    if query_result:
        ret['result'] = True
        ret['comment'] = 'Replication slot {name} exists'.format(name=name)
    elif __opts__['test']:
        ret['changes'] = {
            '{host}.replication_slot.{name}'.format(host=master, name=name):
                'pending create',
        }
        ret['comment'] = 'Replication slot {name} would be created'.format(
            name=name)
    else:
        create_result = __salt__['postgres.psql_query'](
            query=('SELECT pg_create_physical_'
                   "replication_slot('{name}')").format(name=name),
            host=master,
            user=user,
            port=port,
            password=password,
            maintenance_db='postgres')

        if create_result:
            ret['changes'] = {
                '{host}.replication_slot.{name}'.format(
                    host=master, name=name):
                        'created',
            }
            ret['result'] = True
            ret['comment'] = \
                'Replication slot {name} created on {host}'.format(
                    name=name, host=__salt__['pillar.get'](
                        'pg-master',
                        __salt__['pillar.get']('data:pgsync:stream_from')))
        else:
            ret['result'] = False
            ret['comment'] = \
                'Unable to create replication slot {name} on {host}'.format(
                    name=name, host=__salt__['pillar.get'](
                        'pg-master',
                        __salt__['pillar.get']('data:pgsync:stream_from')))

    return ret


def replica_init(name, method, version_major_num=None, server=None):
    """
    Fetch initialized postgresql cluster data directory
    """
    ret = {
        'name': name,
        'result': None,
        'comment': '',
        'changes': {},
    }

    control_path = os.path.join(name, 'global/pg_control')
    if os.path.exists(control_path):
        ret['result'] = True
        ret['comment'] = '{path} exists. Skipping pg-init'.format(
            path=control_path)

        return ret

    cmd_kwargs = {
        'runas': 'postgres',
    }

    if method == 'barman':
        cmd = ('ssh -oStrictHostKeyChecking=no barman-wal-restore@{server} '
               '/usr/bin/sudo -u robot-pgbarman '
               '/usr/local/yandex/barman_restore_last_backup.py '
               '-r {host} -d {path}').format(
                    server=server, host=__grains__['id'], path=name)
    elif method == 'wal-g':
        cmd = ('/usr/bin/envdir /etc/wal-g/envdir '
               'wal-g backup-fetch {path} LATEST').format(path=name)
    elif method == 'basebackup':
        wal_method = 'wal' if version_major_num >= 1000 else 'xlog'
        master = __salt__['pillar.get'](
            'pg-master',
            __salt__['pillar.get']('data:pgsync:stream_from'))
        cmd = ('pg_basebackup --pgdata={path} --checkpoint=fast '
               '--{wal_method}-method=stream --dbname="host={master} '
               'port=5432 dbname=postgres user=repl"').format(
                   path=name, wal_method=wal_method,
                   master=master)
    else:
        ret['result'] = False
        ret['comment'] = 'Unknown method {method}'.format(method=method)
        return ret

    if __opts__['test']:
        ret['changes'] = {
            'pg-init.{path}'.format(path=name): 'pending execution'}
        ret['comment'] = '{cmd} would be executed'.format(cmd=cmd)
    else:
        res = __salt__['cmd.retcode'](cmd, **cmd_kwargs)
        if res != 0:
            ret['result'] = False
            ret['comment'] = '{cmd} exited with {code}'.format(
                cmd=cmd, code=res)
        else:
            ret['changes'] = {'pg-init.{path}'.format(path=name): 'executed'}
            ret['comment'] = '{cmd} exited with {code}'.format(
                cmd=cmd, code=res)
            ret['result'] = True

    return ret


def master_init(name, version):
    """
    Initialize new postgresql cluster
    """
    ret = {
        'name': name,
        'result': None,
        'comment': '',
        'changes': {},
    }

    control_path = os.path.join(name, 'global/pg_control')
    if os.path.exists(control_path):
        ret['result'] = True
        ret['comment'] = '{path} exists. Skipping pg-init'.format(
            path=control_path)

        return ret

    command = 'pg_createcluster --port=5432 {version} data'.format(
        version=version)

    if __opts__['test']:
        ret['changes'] = {
            'pg-init.{path}'.format(path=name): 'pending execution'}
        ret['comment'] = '{command} would be executed'.format(command=command)
    else:
        res = __salt__['cmd.retcode'](command)
        if res != 0:
            ret['result'] = False
            ret['comment'] = '{command} exited with {code}'.format(
                command=command, code=res)
        else:
            ret['changes'] = {'pg-init.{path}'.format(path=name): 'executed'}
            ret['comment'] = '{cmd} exited with {code}'.format(
                cmd=command, code=res)
            ret['result'] = True

    return ret
