#!/usr/bin/env python

import json
import os
import socket
import subprocess
import sys

import porto

try:
    import psycopg2
    import pymongo
except ImportError:
    pass


def get_pg_tier(root, name):
    tier = 'unknown'
    try:
        os.environ['PGPASSFILE'] = \
            '{path}/home/monitor/.pgpass'.format(path=root)
        con = psycopg2.connect(
            ('host={host} port=5432 dbname=postgres '
             'user=monitor connect_timeout=1').format(host=name))
        cur = con.cursor()
        cur.execute('SELECT pg_is_in_recovery()')
        if cur.fetchone()[0]:
            tier = 'replica'
        else:
            tier = 'primary'
    except Exception:
        pass

    return tier


def get_mongodb_tier(root, name):
    tier = 'unknown'
    dbname = 'admin'
    try:
        connect_uri = 'mongodb://{host}:27018/{dbname}'. \
            format(host=name, dbname=dbname)

        conn = pymongo.MongoClient(connect_uri, serverSelectionTimeoutMS=1000)

        tier = 'primary'\
            if conn[dbname].command('ismaster')['ismaster'] else 'secondary'
        conn.close()

    except Exception:
        pass

    return tier


def get_container_tags(itype, geo, name, conn, default_ctype):
    container = conn.Find(name)

    abs_name = container.GetProperty('absolute_name')
    prj = name.split('.')[0][:-1]
    ctype = default_ctype
    root = container.GetProperty('root')

    tier = get_pg_tier(root, name)

    dbaas_file = '{path}/etc/dbaas.conf'.format(path=root)
    if os.path.exists(dbaas_file):
        with open(dbaas_file) as data:
            dbaas = json.load(data)
        prj = str(dbaas['shard_id']).lower()
        ctype = dbaas['cluster_id']
        if dbaas['cluster_type'] == 'mongodb_cluster':
            tier = get_mongodb_tier(root, name)


    result = ('{name} portocontainer={portoname} '
              'a_itype_{itype} a_prj_{prj} '
              'a_ctype_{ctype} a_geo_{geo} '
              'a_tier_{tier}').format(
                  name=name,
                  portoname=abs_name,
                  itype=itype,
                  prj=prj,
                  ctype=ctype,
                  geo=geo,
                  tier=tier)
    return result


def get():
    try:
        with open('/tmp/.grains_conductor.cache') as conductor_cache:
            data = json.load(conductor_cache)
    except Exception:
        data = {}

    hostname = socket.getfqdn()
    itype = 'mdbdom0'
    geo = data.get('short_dc', 'nodc')
    if not geo:
        geo = data.get('switch')[:3]

    if data['primary_vlan'] == '640' or '.mail.yandex.net' in hostname:
        default_ctype = 'pers'
    elif data['primary_vlan'] == '564' or hostname.endswith('.disk.yandex.net'):
        default_ctype = 'disk'
    elif data['primary_vlan'] == '1589' or \
            ('-pgaas-' in hostname and '.db.yandex.net' in hostname):
        default_ctype = 'pgaas'
    else:
        default_ctype = 'test'

    conn = porto.Connection()
    conn.connect()
    for name in conn.List():
        yield get_container_tags(itype, geo, name, conn, default_ctype)


if __name__ == '__main__':
    if os.getuid() != 0:
        subprocess.call(['sudo', sys.argv[0]])
    else:
        for res in get():
            print(res)
