#!/usr/bin/env python
"""
MDB dom0 heartbeat sender
"""

import json
import os
import socket
import subprocess

import requests
from salt.grains import core

DBM_TOKEN = '{{ salt['pillar.get']('data:config:dbm_token', '') }}'
DBM_URL = '{{ salt['pillar.get']('data:config:dbm_url', '') }}'
IGNORE_DISK_FILES = ['lost+found', '.ignore_mounts']
DATACENTERS = {
    'e': 'iva',
    'f': 'myt',
    'h': 'sas',
    'i': 'man',
    'v': 'vla',
}


def get_project(fqdn):
    """
    Guess node project by fqdn
    """
    if '-pgaas-' in fqdn:
        return 'pgaas'
    if '-test-' in fqdn:
        return 'sandbox'
    if 'disk.yandex.net' in fqdn:
        return 'disk'
    return 'pers'


def get_memory():
    """
    Get memory info from /proc filesystem
    """
    with open('/proc/meminfo', 'r') as meminfo:
        for line in meminfo.readlines():
            if line.startswith('MemTotal:'):
                _, memory_kb, _ = line.split()

    memory = int(memory_kb) * 1024
    # Overcommit is forbidden, 3Gb are reserved for host system.
    memory -= 3 * (1024 * 1024 * 1024)
    return memory


def get_disk_info():
    """
    Extract disk info from /etc/server_info.json
    """
    ssd_space = 0
    sata_space = 0
    max_io = 0

    with open('/etc/server_info.json', 'r') as server_info:
        info = json.load(server_info)
        for mountpoint in info['points_info']:
            device = info['points_info'][mountpoint]['device']
            first_disk = info['points_info'][mountpoint]['disks_info'][0]
            for key in first_disk.keys():
                disk_type = first_disk[key]['disk_type']

            proc = subprocess.Popen(['df', device], stdout=subprocess.PIPE)
            output = proc.communicate()[0]
            size_kb = output.split('\n')[1].split()[1]
            size = int(size_kb) * 1024

            num_disks = len(info['points_info'][mountpoint]['disks_info'])

            if disk_type == 'hdd':
                sata_space += size
                max_io += 25 * num_disks
            else:
                ssd_space += size
                max_io += 250 * num_disks

    if ssd_space > 0:
        ssd_space -= 10 * 1024 * 1024 * 1024
    if sata_space > 0:
        sata_space -= 10 * 1024 * 1024 * 1024
    max_io *= 1024 * 1024

    return ssd_space, sata_space, max_io


def get_disks():
    """
    Get raw disks (/disks/<uuid>)
    """
    ret = []

    if os.path.exists('/disks'):
        with open('/proc/mounts') as proc_mounts:
            mounts = [x.split() for x in proc_mounts]
        for disk_id in os.listdir('/disks'):
            full_path = os.path.join('/disks', disk_id)
            if not os.path.exists(os.path.join(full_path, '.ignore_mounts')):
                device = None
                for mount in mounts:
                    if mount[1] == full_path:
                        device = mount[0]
                        break
                if not device:
                    continue
            stat = os.statvfs(full_path)
            size = stat.f_bsize * stat.f_blocks
            has_data = bool([
                x for x in os.listdir(full_path) if x not in IGNORE_DISK_FILES
            ])
            ret.append({
                'id': disk_id,
                'max_space_limit': size,
                'has_data': has_data,
            })

    return ret


def get_net_speed():
    """
    Get network info from salt grains and ethtool
    """
    iface_names = []
    ip6_interfaces = core.ip6_interfaces()['ip6_interfaces']
    for iface in ip6_interfaces:
        if 'eth' in iface and ip6_interfaces[iface]:
            iface_names.append(iface)
    if len(iface_names) == 1:
        iface = iface_names[0]

    proc = subprocess.Popen(['ethtool', iface], stdout=subprocess.PIPE)
    output = proc.communicate()[0]
    for line in output.splitlines():
        if 'Speed:' in line:
            speed_mbitsps = int(line.split()[-1].replace('Mb/s', ''))

    net_speed = speed_mbitsps * 1024 * 1024 / 8
    return net_speed


def heartbeat(fqdn, data):
    """
    Call DBM backend via REST API to report heartbeat
    """
    url = '{url}/api/v2/dom0/{fqdn}'.format(url=DBM_URL, fqdn=fqdn)
    headers = {
        'Content-Type': 'application/json',
        'Authorization': 'OAuth {token}'.format(token=DBM_TOKEN),
    }
    res = requests.post(
        url, headers=headers, data=json.dumps(data), verify=False, timeout=5)
    if res.status_code != 200:
        raise Exception('Request to DBM failed ({code}: {text})'.format(
            code=res.status_code, text=res.text))


def porto_alive():
    """
    Check if porto daemon is alive
    """
    import porto
    conn = porto.Connection()
    conn.connect()
    _ = conn.ListContainers()


def _main():
    fqdn = socket.gethostname()
    geo = DATACENTERS.get(fqdn.split('.')[0][-1], 'all')
    num_cpus = os.sysconf('SC_NPROCESSORS_ONLN')
    ssd_space, sata_space, max_io = get_disk_info()
    net_speed = get_net_speed()

    generation = 1
    if num_cpus >= 56 and net_speed >= 1310720000:
        generation = 2

    data = {
        'project': get_project(fqdn),
        'geo': geo,
        'cpu_cores': num_cpus,
        'memory': get_memory(),
        'ssd_space': ssd_space,
        'sata_space': sata_space,
        'max_io': max_io,
        'net_speed': net_speed,
        'generation': generation,
        'heartbeat': True,
        'disks': get_disks(),
    }

    if os.path.exists('/etc/dbm_heartbeat_override.json'):
        with open('/etc/dbm_heartbeat_override.json') as override:
            data.update(json.load(override))

    porto_alive()
    heartbeat(fqdn, data)


if __name__ == '__main__':
    _main()
