#!/usr/bin/env python
"""
Script for moving container between dom0 hosts
"""

import argparse
import json
import os
import subprocess
import time
from ConfigParser import ConfigParser

import requests

import porto
import salt.client

PORTO_CONN = porto.Connection()
PORTO_CONN.connect()
SALT_CLIENT = salt.client.Caller()

DOM0_SALT_CMD = ('salt-call state.sls components.dom0porto.containers '
                 'queue=True --state-output=changes -l quiet')
CONT_SALT_CMD = (r"sed -i -e 's/.*yandex\.net$//g' -e '/^\s*$/d' /etc/hosts",
                 'salt-call saltutil.sync_all', 'salt-call state.highstate '
                 'queue=True --state-output=changes -l quiet')


def parse_args():
    """
    Function for parsing arguments
    """
    parser = argparse.ArgumentParser(description=__doc__)

    parser.add_argument('container', help='container to move')
    parser.add_argument('new_dom0', help='dom0-host where to move')

    return parser.parse_args()


def get_config():
    """
    Simple fuction to read ~/.mdb_move_container.conf
    """
    config = ConfigParser()
    config.read(os.path.expanduser('~/.mdb_move_container.conf'))
    return config


def get_paths(container, config):
    """
    Query salt to get container volume paths
    """
    volumes = SALT_CLIENT.cmd('pillar.get', 'data:porto:%s:volumes' % container)
    return [i['dom0_path'] for i in volumes]


def set_downtime(fqdn, config, duration=8 * 3600):
    """
    Set downtime via juggler-api
    """
    headers = {
        'Authorization':
            'OAuth {token}'.format(token=config.get('juggler', 'token')),
        'Content-Type':
            'application/json',
    }
    now = int(time.time())
    end = now + int(duration)
    data = json.dumps({
        'end_time': end,
        'start_time': now,
        'description': 'container move',
        'filters': [{
            'host': fqdn,
            'namespace': 'disk-db',
        }],
    })
    resp = requests.post(
        config.get('juggler', 'url'),
        headers=headers,
        data=data,
        verify=False,
    )
    if resp.status_code != 200:
        raise Exception('Could not set downtime ({code}: {text})'.format(
            code=resp.status_code, text=resp.text))


def remote_mkdir(dom0, path):
    """
    Create directory on remote host
    """
    cmd = [
        'ssh',
        'root@{dom0}'.format(dom0=dom0),
        'mkdir -p {path}'.format(path=path),
    ]
    code = subprocess.Popen(cmd).wait()
    if code != 0:
        raise Exception('Could not create directory {dir} on host '
                        '{dom0}'.format(dir=path, dom0=dom0))


def run_state(dom0):
    """
    Run salt state on dom0 host
    """
    cmd = ['ssh', 'root@{dom0}'.format(dom0=dom0), DOM0_SALT_CMD]
    code = subprocess.Popen(cmd).wait()
    if code != 0:
        raise Exception('Could not run highstate on {dom0}'.format(dom0=dom0))


def run_state_in_container(dom0, fqdn):
    """
    Run salt highstate inside porto container
    """
    for cont_cmd in CONT_SALT_CMD:
        cmd = [
            'ssh',
            'root@{dom0}'.format(dom0=dom0),
            'portoctl shell {fqdn} {salt}'.format(fqdn=fqdn, salt=cont_cmd),
        ]
        code = subprocess.Popen(cmd).wait()
        if code != 0:
            raise Exception(
                'Could not run highstate in {fqdn}'.format(fqdn=fqdn))


def rsync_dir(dom0, path):
    """
    Rsync local dir to remote host
    """
    cmd = [
        'rsync',
        '-a',
        '--numeric-ids',
        '{path}/'.format(path=path),
        'root@{dom0}:{path}/'.format(dom0=dom0, path=path),
    ]
    code = subprocess.Popen(cmd).wait()
    if code != 0:
        raise Exception('Could not rsync {dir} on host '
                        '{dom0}'.format(dir=path, dom0=dom0))


def _main():
    args = parse_args()
    config = get_config()
    paths = get_paths(args.container, config)
    print('Setting downtime')
    set_downtime(args.container, config)
    print('Creating paths')
    for path in paths:
        remote_mkdir(args.new_dom0, path)

    print('Stopping container')
    container = PORTO_CONN.Find(args.container)
    if not container:
        raise Exception('Could not find container {container}'.format(
            container=args.container))
    container.Stop()

    for path in paths:
        print('Rsyncing {path}'.format(path=path))
        rsync_dir(args.new_dom0, path)

    print('Please move container in salt repo manually and wait for two minutes after push')
    print('Waiting...')
    answer = raw_input('Are you ready to continue? (say "yes")\n')
    while True:
        if not answer.strip():
            answer = raw_input()
            continue
        if answer.strip() != "yes":
            print("Exit")
            return
        else:
            break

    print('Running highstate on new dom0 host')
    run_state(args.new_dom0)

    print('Running highstate in container')
    run_state_in_container(args.new_dom0, args.container)

    print('Destroying container')
    container.Destroy()
    for path in paths:
        new_path = '{path}.{ts}.bak'.format(path=path, ts=int(time.time()))
        os.rename(path, new_path)


if __name__ == '__main__':
    _main()
