#!/usr/bin/env python

import pymongo
from pymongo.uri_parser import parse_uri
import argparse
import sys

class MongoDB(object):
    def __init__(self, uri=None, db=None):
        self.con = self.__connect(uri, db)
        self.db = db
        self.uri = uri
        self.con_opts = parse_uri(self.uri)
        # con_opts = 
        # { 
        #   'username': 'admin', 
        #   'nodelist': [('localhost', 27017)], 
        #   'database': 'dv', 
        #   'collection': None, 
        #   'password': 'pw', 
        #   'options': {'authsource': 'admin'}
        # }

    def __connect(self, uri, db='admin'):
        return pymongo.MongoClient(
                uri,
                socketTimeoutMS=10000,
                connectTimeoutMS=10000,
                serverSelectionTimeoutMS=10000,
                waitQueueTimeoutMS=10000,
                )[db]

    def validate_user(self, name=None, pwd=None, roles=[]):
        """
        Try connecting using supplied credentials. 
        If successfull, list all privileges and compare against provided list.
        
        Return True if user does not require changes and False otherwise.
        """
        for host, port in self.con_opts['nodelist']:
            user_uri = 'mongodb://{user}:{pw}@{host}:{port}/{db}?authSource={db}&authMechanism={mech}'.format(
                user=name,
                pw=pwd,
                host=host,
                port=port,
                db=self.db,
                mech=self.con_opts.get('options', {}).get('mechanism', 'SCRAM-SHA-1'),
            )
            try:
                user_con = self.__connect(user_uri, self.db)
                auth_data = user_con.command({'usersInfo': name})['users']
                # auth_data = [
                #    {
                #       u'_id': u'docviewer.dv',
                #       u'db': u'docviewer',
                #       u'roles': [{u'db': u'docviewer', u'role': u'dbAdmin'},
                #                 {u'db': u'docviewer', u'role': u'readWrite'}],
                #       u'user': u'dv'
                #    }
                # ]
            except pymongo.errors.OperationFailure as err:
                if err.code == 18:  # {u'code': 18, u'ok': 0.0, u'errmsg': u'Authentication failed.'}
                    return False
                else:
                    raise
            for user in auth_data:
                user_roles = [x['role'] for x in user['roles'] if x['db'] == self.db]
                if user['user'] == name and sorted(user_roles) == sorted(roles):
                    return True
            return False

    def create_user(self, name=None, pwd=None, roles=[]):
        """
        Create or modify user.
        """
        return self.con.add_user(
                name,
                password=pwd,
                roles=roles,
                )
    
if __name__ == '__main__':
    
    arg = argparse.ArgumentParser(description="""
            MongoDB user creator
            """
            )   
    arg.add_argument('-r', '--roles', required=True, nargs='+', metavar='<str>',
            help='add these roles')
    arg.add_argument('-d', '--db', required=False, nargs='+', type=str, default='admin', metavar='<str>',
            help='apply access to these databases')
    arg.add_argument('-u', '--user', required=True, type=str, metavar='<str>',
            help='username to add')
    arg.add_argument('-p', '--password', required=True, type=str, metavar='<str>',
            help='username to add')
    arg.add_argument('-c', '--uri', required=True, type=str, metavar='<str>',
            help='URI to use when connecting')
    arg.add_argument('-t', '--test', required=False, action='store_true', default=False,
            help='Do not touch database, merely detect if changes required')
    s = vars(arg.parse_args())

    try:
        for db_name in s.get('db', []):
            db = MongoDB(uri=s.get('uri'), db=db_name)
            try:
                params = {
                    'name': s.get('user', 'UNDEFINED_USER'),
                    'pwd': s.get('password', ''),
                    'roles': s.get('roles', []),
                }
                user_valid = db.validate_user(**params)
                if user_valid:
                    continue
                if s.get('test', 'True'):
                    print 'changed=yes comment="{u} needs to be modified or created"'.format(u=s.get('user'))
                    continue
                db.create_user(**params)
                if db.validate_user(**params):
                    print 'changed=yes comment="{u} created or modified successfully"'.format(u=s.get('user'))
                else:
                    print 'changed=yes comment="{u} wasnt modified but no error was hit"'.format(u=s.get('user'))
                    sys.exit(1)
            except pymongo.errors.NotMasterError:
                pass
    except pymongo.errors.ServerSelectionTimeoutError:  # mongo{s,db} is probably dead. Changes will light up as soon as it gets up anyway.
        sys.exit(0)
    except pymongo.errors.OperationFailure as err:
        print 'changed=no comment="{e}"'.format(e=err.details['errmsg'])
        sys.exit(1)
    except Exception as exc:
        print 'changed=no comment="{e}"'.format(e=exc)
        sys.exit(1)
