#!/usr/bin/env python3.5
# -*- coding: utf-8 -%-
"""
Check that ClickHouse backup is not too old.
"""

import argparse
import json
import logging
import subprocess
import sys
from datetime import datetime, timezone


def main():
    """
    Program entry point.
    """
    logging.basicConfig(format='%(message)s')

    args_parser = argparse.ArgumentParser()
    args_parser.add_argument(
        '-w', '--warn',
        type=int,
        default=2,
        help='Warn if backup is older than this many days')
    args = args_parser.parse_args()

    try:
        result = run('sudo ch-backup list')
        if not str.strip(result):
            die(1, 'No backups found')

        result = run('sudo ch-backup show LAST')
        backup = json.loads(result)
        meta = backup['meta']

    except Exception as exc:
        die(1, 'Failed to get information about the latest backup: %s' % str(exc))

    try:
        backup_time = datetime.strptime(meta['end_time'], meta['date_fmt'])
        if backup_time.tzinfo is None:
            backup_time = backup_time.replace(tzinfo=timezone.utc)
        delta = datetime.now(timezone.utc) - backup_time

        days = delta.days
        if days >= args.warn:
            die(1, 'Latest backup was more than %s days ago' % days)

        die(0, 'OK')

    except Exception as exc:
        logging.exception('Caught exception:')
        die(1, 'Unable to get status: %s' % repr(exc))


def run(cmd):
    """
    Run the command and return its output.
    """
    result = subprocess.run(cmd, shell=True, check=True,
                            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    return result.stdout.decode()


def die(status, message):
    """
    Emit status and exit.
    """
    print('%s;%s' % (status, message))
    sys.exit(0)


if __name__ == '__main__':
    main()
