#!/usr/bin/env python3.5

import argparse
import socket
import sys
import time

import requests


def main():
    """
    Program entry point.
    """
    args_parser = argparse.ArgumentParser()
    args_parser.add_argument(
        '-n', '--number',
        type=int,
        default=3,
        help='The max number of retries.')
    args_parser.add_argument(
        '-p', '--port',
        type=int,
        help='ClickHouse HTTP(S) port to use.')
    args_parser.add_argument(
        '-s', '--ssl',
        action="store_true",
        help='Use HTTPS rather than HTTP.')
    args_parser.add_argument(
        '--ca_bundle',
        help='Path to CA bundle to use.')

    args = args_parser.parse_args()

    host = socket.gethostname()
    if args.port:
        port = args.port
    else:
        port = 8443 if args.ssl else 8123

    error = 'unexpected error'
    for n in range(args.number):
        if n != 0:
            time.sleep(1)

        try:
            if request(host, port, args.ssl, args.ca_bundle) != 'Ok.':
                continue

            if request(host, port, args.ssl, args.ca_bundle, 'SELECT 1') != '1':
                continue

            die(0, 'OK')

        except Exception as e:
            error = repr(e)
            time.sleep(1)

    die(2, 'ClickHouse is dead ({0})'.format(error))


def request(host, port, ssl, ca_bundle, query=None):
    """
    Send request to ClickHouse.
    """
    protocol = 'https' if ssl else 'http'
    verify = ca_bundle if ca_bundle else ssl
    params = {}
    if query:
        params['query'] = query

    r = requests.get(
        '{0}://{1}:{2}'.format(protocol, host, port),
        params=params, timeout=0.1, verify=verify)
    r.raise_for_status()
    return r.text.strip()


def die(status, message):
    """
    Emit status and exit.
    """
    print('%s;%s' % (status, message))
    sys.exit(0)


if __name__ == '__main__':
    main()
