#!/usr/bin/env python3.5

import argparse
import socket
import sys
import time

import requests
import yaml


def main():
    """
    Program entry point.
    """
    try:
        config = get_config()
        do_check(config)
    except Exception as e:
        die(2, 'Error: ' + repr(e))

    die(0, 'OK')


def get_config():
    """
    Parse command-line arguments and return config.
    """
    parser = argparse.ArgumentParser()
    parser.add_argument('-c', '--config_file', help='Config file.')
    args = parser.parse_args()

    with open(args.config_file, 'r') as f:
        return yaml.load(f)


def do_check(config):
    """
    Perform the check.
    """
    metrics = get_metrics(config)
    check_metrics(metrics, config)


def get_metrics(config):
    """
    Select and return metrics form system.replicas.
    """
    ssl = config['ssl']
    ca_bundle = config['ca_bundle']
    url = '{protocol}://{host}:{port}'.format(
        protocol='https' if ssl else 'http',
        host=socket.gethostname(),
        port=8443 if ssl else 8123)
    verify = ca_bundle if ca_bundle else ssl
    timeout = config['timeout']
    query = 'SELECT database, table, future_parts, parts_to_check, queue_size,' \
            ' inserts_in_queue, merges_in_queue FROM system.replicas FORMAT JSON'

    error = 'unexpected error'
    for _ in range(config['retries']):
        try:
            r = requests.get(
                url, params={'query': query},
                verify=verify, timeout=timeout)
            return r.json()['data']
        except Exception as e:
            error = repr(e)
            time.sleep(1)
    die(2, 'Failed to get metrics ({0})'.format(error))


def check_metrics(metrics, config):
    """
    Check that metrics are within acceptable levels.
    """
    thresholds_conf = config['triggers']
    default_thresholds = thresholds_conf['default']
    status_map = {'crit': 2, 'warn': 1}

    status = 0
    triggers = []
    for row in metrics:
        db_table = '{}.{}'.format(row['database'], row['table'])
        table_thresholds = thresholds_conf.get(db_table, {})

        for key, value in row.items():
            if key not in default_thresholds:
                continue

            thresholds = table_thresholds.get(key, default_thresholds[key])

            report = ''
            for prior in 'crit', 'warn':
                threshold = thresholds[prior]
                if value > threshold:
                    report += '{}: {} {} > {} ({});'.format(db_table, key, value, threshold, prior)
                    triggers.append((status_map[prior], report))
                    break

    if triggers:
        triggers.sort(reverse=True, key=lambda x: x[0])
        message = ' '.join(x[1] for x in triggers)
        die(status, message)


def die(status, message):
    """
    Emit status and exit.
    """
    print('%s;%s' % (status, message))
    sys.exit(0)


if __name__ == '__main__':
    main()
