#!/bin/bash

debug(){
   if [ "$_debug" ]; then
      lineno=`caller 0`
      echo "`printf "line #%-4s %-12s" ${lineno% *}`: $@"
   fi
}

setlevel() {
    if [ ${level:-0} -lt $1 ]
    then
        level=$1
    fi
}

die() {
    IFS=';'
    echo "${level:-0};${msg[*]:-OK}"
    exit 0
}

usage() {
cat<<HELP
SYNOPSIS
  $0  [ OPTIONS ]
OPTIONS
  -d
      Debug mode

  -s
      Use sudo. Monitoring mode.

  -h
      Print a help message.

HELP
    exit 0
}

while getopts "i:sdh" OPTION
do
    case $OPTION in
        i)
            INTERFACE="$OPTARG"
        ;;
        s)
            USE_SUDO="1"
        ;;
        d)
            _debug=1
        ;;
        h)
            usage && exit 1
            ;;
        ?)
            usage && exit 1
            ;;
    esac
done

if [ -f /usr/local/sbin/autodetect_environment ] ; then
    is_virtual_host=0
    . /usr/local/sbin/autodetect_environment >/dev/null 2>&1 || true
    if [ "$is_virtual_host" -eq 1 ] ; then
        debug "Is virtual host. Skip check"
        setlevel 0; msg[m++]='OK'; die
    fi
fi

if cat /proc/cpuinfo | grep 'model name' | grep -q 'KVM'
then
    debug "Is virtual host. Skip check"
    setlevel 0; msg[m++]='OK'; die
elif sudo grep /var/log/dmesg -e 'Booting paravirtualized kernel on KVM' -q 2>/dev/null
then
    debug "Is virtual host. Skip check"
    setlevel 0; msg[m++]='OK'; die
fi

# Getting interface name
if [ ! -z "$INTERFACE" ]
then
    interfaces=$INTERFACE;
else
    if [ ! -z "$USE_SUDO" ]
    then
        interfaces=`sudo ip link list 2>/dev/null| grep 'state UP' |grep -v '@' | awk  -F ':' '{print $2}' |xargs`;
    else
        interfaces=`ip link list 2>/dev/null| grep 'state UP' |grep -v '@' | awk  -F ':' '{print $2}' |xargs`;
    fi
fi

for interface in $interfaces
do
    debug "Interface: $interface"
    # Check for valid interface name
    echo $interface | egrep -qo 'bond[0-9]+' && { debug "Unsupported iface $interface"; continue; }
    echo $interface | egrep -qo 'eth[0-9]+' || { debug "Wrong iface name $interface"; continue; }
    
    maxspeed=10000;
    speed=$(sudo /sbin/ethtool $interface 2>/dev/null| awk '/Speed/ {print $2}' | cut -dM -f 1)
    
    if [ "$speed" -lt "${maxspeed}" ]; then
        setlevel 1; msg[m++]="Bad link speed $speed for iface $interface";
    else
        msg[m++]="Link speed for iface $interface is OK";
    fi
done

if [ -z "$level" ]
then
    unset msg
fi

die
exit 0;
