#!/usr/bin/env python3.5
# -*- coding: utf-8 -%-
"""
Check that genbackup backup exists and  is not too old.
"""

import argparse
import json
import logging
import subprocess
import sys
from datetime import datetime


DEFAULT_DATE_FMT = '%Y-%m-%d %H:%M:%S'


def main():
    """
    Program entry point.
    """
    logging.basicConfig(format='%(message)s')

    args_parser = argparse.ArgumentParser()
    args_parser.add_argument(
        '--warn-old',
        type=int,
        default=2,
        help='Warn if backup is older than this many days')
    args_parser.add_argument(
        '--warn-count',
        type=int,
        default=10,
        help='Warn if backup count is greater than given threshold')
    args = args_parser.parse_args()

    try:
        retcode, stdout, stderr = run('sudo genbackup list')
        if retcode:
            die(2, 'Can not get backup list: exit code {0}'.format(retcode))

        if not str.strip(stdout):
            die(2, 'No backups found')

        backups = stdout.split('\n')

        if len(backups) > args.warn_count:
            die(1, 'Too many backups exist: {0} > {1}'.format(len(backups), args.warn_count))

        latest = backups[0]
        retcode, stdout, stderr = run('sudo genbackup show -p {0}'.format(latest))
        if retcode:
            die(2, 'Can not get last backup meta')
        backup = json.loads(stdout)
        backup_time = datetime.strptime(backup['dump_finish_time'], DEFAULT_DATE_FMT)
        delta = datetime.utcnow() - backup_time
        days = delta.days
        if days >= args.warn_old:
            die(1, 'Latest backup was more than {0} days ago: {1}'.format(days, latest))

        die(0, 'OK')

    except ValueError as exc:
        die(2, 'Can not load load last backup meta ({0}): {1}'.format(latest, repr(exc)))
    except KeyError as exc:
        die(2, 'Can not get finish time: {0}'.format(repr(exc)))
    except Exception as exc:
        die(2, 'Unable to get status: {0}'.format(repr(exc)))


def run(cmd):
    """
    Run the command and return its output.
    """
    result = subprocess.run(cmd, shell=True,
                            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    return result.returncode, result.stdout.decode(), result.stderr.decode()


def die(status, message):
    """
    Emit status and exit.
    """
    print('{0};{1}'.format(status, message))
    sys.exit(0)


if __name__ == '__main__':
    main()

