#!/bin/bash
#
# Provides: cron

# Environment
me=${0##*/}     # strip path
me=${me%.*}     # strip extension
prefix=${HOME}/agents
conf=${prefix}/etc/${me}.conf

# get config
[ -e "${conf}" ] && . "${conf}"

# default values
run_path=${RUN_PATH:-/var/run/statbox}
max_instances=${MAX_INSTANCES:-1}
daemon=${DAEMON:-$me}
hostname=${HOSTNAME:-`hostname`}

die () {
    printf "PASSIVE-CHECK:$me;%s;%s\n" "$1" "$2"
    exit 0
}

check_proc_linux()
{
    # 0. defaults
    local current_instances process pid search_mask
    process=$1
    search_mask='[k]afka.Kafka'
    current_instances=0
    pid=''
    ### TODO: sometimes pidfile names do not end with 'd'!
    pidfile="${run_path}/statbox-kafka.pid" 
    # 1. check by pidfile.
    if [ -s "${pidfile}" ]; then
          # read pid
          pid=$(cat "${pidfile}")
          # check that pid
          ps --pid ${pid} &>/dev/null || die 2 "${process}: stale pidfile present"
    else
          # No pidfile!
          die 2 "${process}: no pidfile found"
    fi
    
    # 2. Count instances, taking LXC namespace separation into consideration
    # pgrep doesnt know any difference between, e.g. CROND and crond, and -f is too risky.
    if [ -e '/proc/self/cpuset' ]; then 

          own_domain=`cat /proc/self/cpuset` # e.g.: /
          for proc_pid in `pgrep -P 1 -f "${search_mask}"`
          do
              [ "x${own_domain}" = x`cat /proc/${proc_pid}/cpuset` ] && current_instances=$((${current_instances}+1))
          done
    else
          current_instances=`pgrep -P 1 -f "^${process}" | wc -l`
    fi
    [ ${current_instances} -gt ${max_instances} ] && die 1 "${process}: abnormal process count: ${current_instances} (max ${max_instances})"
    [ ${current_instances} -eq 0 ] && die 2 "${process}: not running"
    die 0 "${process}: ${current_instances} procs running"
}

check_proc_posix()
{
    return 1
    # TODO
}

case `uname` in
    Linux)          
          check_proc_linux "${daemon}"
          ;;
    *)          
          die 1 "${process}: OS `uname` is not supported"
          ;;
esac
