#!/usr/bin/python

import httplib
from pprint import pprint
import sys,socket,re
import json
import argparse

class LogbrokerException(Exception):
    pass

class LogbrokerOffsets(object):
    def __init__(self, timeout=10.0, port=8999):
        self.timeout = timeout
        self.port = port
    def __get(self, host, url, data=None):
        c = False
        r = False
        try:
            socket.setdefaulttimeout(self.timeout)
            c = httplib.HTTPConnection(host=host, port=self.port, timeout=self.timeout)
            # Perform request.
            c.request('GET', url, data)
            # Fetch the response headers.
            r = c.getresponse()
            # Read the response body.
            raw_data = r.read()
            parsed_data = self.__parse_response(raw_data)
        except Exception as e:
            raise LogbrokerException('general error, "%s"' % unicode(e))
        except httplib.HTTPException as e:
            raise LogbrokerException('transport error, "%s"' % unicode(e))

        if r.status != httplib.OK:
            raise LogbrokerException('request error, status %s, response "%s"' % (
                r.status, 
                raw_data
                )
            )

        return parsed_data

    def __parse_response(self, data):
        info = {}
        for line in data.splitlines():
            try:
                topicpart, offset, logstart, logsize, lag, owner = line.split('\t')
                info[topicpart] = {
                        'offset': offset,
                        'logstart': logstart,
                        'logsize': logsize,
                        'lag': lag,
                        'owner': owner
                        }
            except Exception:
                # Dont care about junk.
                pass
        return info

    def __get_state(self, ident=None, client=None, logtype=None, dc=''):
        filter = 'log-type={lt}'.format(lt=logtype) if logtype else 'ident={i}'.format(i=ident)
        url = '/pull/offsets?{filter}&client={client}'.format(filter=filter, client=client)
        dc_subdomain = '%s.' % dc if dc else ''
        offsets = self.__get(url=url, host='{dc}logbroker.yandex.net'.format(dc=dc_subdomain))
        # Logbroker puts irrelevant topics ("other") into the output 
        # if it is unable to find the requested ident or logtype.
        # Filter it out.
        state = {}
        for tp, info in offsets.iteritems():
            if logtype in tp and tp.startswith('rt3.{dc}'.format(dc=str(dc))):
                state.update({tp: info})
        return state
        # return {tp:info for (tp,info) in offsets.iteritems() if logtype in tp and tp.startswith('rt3.{}'.format(str(dc)))}
    
    def get_owners(self, ident=None, client=None, dc=''):
        state = self.__get_state(ident=ident, client=client, dc=dc)
        owners = []
        for tp, data in state.iteritems():
            owner = data['owner'].strip()
            if owner != 'none': owners.append(owner)
        return owners

    def get_total_lag(self, dc='', **kwargs):
        state = self.__get_state(dc=dc, **kwargs)
        total_lag = 0
        for tp, data in state.iteritems():
            total_lag += int(data['lag'])
        return total_lag
    
    def get_total_offsets(self, dc='', **kwargs):
        state = self.__get_state(dc=dc, **kwargs)
        total_offsets = 0
        for tp, data in state.iteritems():
            total_offsets += int(data['offset'])
        return total_offsets


def write_state_file(owners):
    try:
        f = open('/tmp/logbroker_session_owners', 'w')
        f.write(json.dumps(owners))
        f.close()
    except Exception:
        return False
    
    return True

def read_state_file():
    state = []
    try:
        f = open('/tmp/logbroker_session_owners', 'r')
        state = json.loads(''.join(f.readlines()))
    finally:
        return state

def print_graphite(num, logtype='unknown', client='unknown', mtype='lags'):
    import time
    desc = 'mail.mail_logconsumer.{hostname}.{tp}.{logtype}.{client}'.format(
            hostname=socket.getfqdn().replace('.','_'),
            tp=mtype,
            logtype=logtype,
            client=client
            )
    print '{description} {number} {time}'.format(description=desc, number=num, time=int(time.time()))

def print_juggler(code, desc):
    print '{code};{desc}'.format(code=code, desc=desc)
    sys.exit(0)

def get_local_dc():
    def __read_dcfile():
        dc = False
        try:
            dc = open('/tmp/dc_affinity').read().strip()
        except Exception:
            return False
        return dc
    def __write_dcfile(dc):
        try:
            f = open('/tmp/dc_affinity', 'w')
            f.write(dc)
            f.close()
        except Exception:
            pass

    def __call_golem():
        dc = False
        try:
            c = httplib.HTTPConnection(host='ro.admin.yandex-team.ru')
            c.request('GET','/api/host_query.sbml?hostname={host}&columns=short_dc'.format(host=socket.getfqdn()))
            r = c.getresponse()
            dc = r.read().strip()
        except Exception:
            pass
        return dc

    # Try reading a file, if failed, then request conductor.
    dc = __read_dcfile()
    if not dc:
        dc = __call_golem()

    if not re.search('^[a-z]{3}', str(dc)):
        return False
    # Save result for the future use.
    __write_dcfile(dc)
    return dc

if __name__ == '__main__':
    arg = argparse.ArgumentParser(description="""
            Juggler plugin to check for consumption lag length.
            """
            )
    arg.add_argument('-i', '--ids', action='append', required=True, metavar='<logtype>:<client>',
            help='Logtype and clientid to check. Can be given multiple times.')
    arg.add_argument('-d', '--dc', metavar='<dc>', default=get_local_dc(),
            help='Override dc affinity. Default is autodetect.')
    arg.add_argument('-g', '--graphite', action='store_true', default=False,
            help='Emit graphite statistics.')
    
    settings = vars(arg.parse_args())
    
    ids = [id.split(':') for id in settings.get('ids', [])]
    
    # Cannot continue if local dc is not known
    if not settings.get('dc', False): print_juggler(1, 'unable to determine dc affiliation')
    # Cannot continue if no clientid or ident is provided either.
    if not settings.get('ids', False): print_juggler(1, 'no client ids are given')
    
    # Create logbroker object.
    lb = LogbrokerOffsets()
    lag = 0
    offset = 0
    for (logtype,client) in ids:
        try:
            lag += lb.get_total_lag(logtype=logtype, client=client, dc=settings.get('dc'))
            offset += lb.get_total_offsets(logtype=logtype, client=client, dc=settings.get('dc'))
            if settings.get('graphite'):
                print_graphite(lag, logtype, client, mtype='lags')
                print_graphite(offset, logtype, client, mtype='offsets')
        except NotImplementedError as exc:
            print_juggler(1, 'exception: {e}'.format(e=exc))

    if not settings.get('graphite'): print_juggler(0, lag)
