#!/usr/bin/env python

import pymongo
import sys
import argparse
import socket
from collections import namedtuple
from pprint import pprint

Result = namedtuple('Result', 'code, message')
RS_STATE = '/tmp/mongodb_monitoring_rs'
PROFILER_STATE = '/tmp/mongodb_monitoring_slowlog'

class Status(object):
    """ Class for holding Juggler status """
    code = 0
    text = []

    def set_code(self, new_code):
        """ Set the code if it is greater than the current. """
        if new_code > self.code:
            self.code = new_code

    def append(self, new_text):
        """Accumulate the status text"""
        self.text.append( new_text )

    def report(self, code=0, message=None):
        """ Output formatted status message"""
        # concatenate all received statuses
        if message is None:
            message = '. '.join(self.text)
        if not message and self.code == 0:
            message = 'ok'
        # Check if code is above current setting
        self.set_code(code)
        # strip underscores and newlines.
        print '%d;%s' % (self.code, message.replace('_', ' ').replace('\n','').replace('mail.yandex.net', 'm'))
        sys.exit(0)

class CheckMongoDB(object):
    def __init__(self, uri=None, db=None):
        self.con = self.__connect(uri)
        self.db = db
        self.uri = uri

    def __connect(self, uri):
        return pymongo.MongoClient(
                uri,
                socketTimeoutMS=5000,
                connectTimeoutMS=5000,
                serverSelectionTimeoutMS=5000,
                waitQueueTimeoutMS=5000,

                )['admin']

    def locked_queue(self, crit=None, warn=None):
        code = 0
        if crit is None:
            crit = 20
        if warn is None:
            warn = 10
        
        status = self.con.command('serverStatus', metrics=0, recordStats=0, locks=1)
        active = status['globalLock']['currentQueue']['total']
        
        if active >= crit:
            code = 2
        elif active >= warn:
            code = 1
        else:
            code = 0
        return Result(code=code, message='%d reqs' % active)
    
    def lag(self, crit=None, warn=None):
        code = 0
        if crit is None:
            crit = 100
        if warn is None:
            warn = 20
        status = self.con.command('replSetGetStatus')
        current_instance = {}
        primary = {}
        # Get shortcuts.
        for state in status['members']:
            if 'self' in state.keys():
                current_instance = state
            if state['stateStr'] == 'PRIMARY':
                primary = state
        
        if current_instance['stateStr'] == 'PRIMARY':
            return Result(code=0, message='OK')

        if current_instance.get('errmsg') is not None:
            return Result(code=2, message=current_instance.get('errmsg'))

        diff = primary['optimeDate'] - current_instance['optimeDate']
        diff_secs = (diff.days * 86400 + diff.seconds)
        if diff_secs > 1500000000:
            return Result(code=1, message='initial sync in progress')
        if diff_secs >= crit:
            code = 2
        elif diff_secs >= warn:
            code = 1
        return Result(code=code, message='%d sec' % diff_secs)
    def up(self, crit=None, warn=None):
        status = self.con.command('ping')
        return Result(code=0, message='OK')

    def master(self, crit=None, warn=None):
        status = self.con.command('replSetGetStatus')
        primaries_cnt = 0
        for state in status['members']:
            if state['stateStr'] == 'PRIMARY':
                primaries_cnt += 1
        
        if primaries_cnt > 1:
            return Result(code=2, message="%d primaries in rs" % primaries_cnt)
        elif primaries_cnt < 1:
            return Result(code=2, message="no primaries in rs")
        return Result(code=0, message='OK')

    def role(self, crit=None, warn=None):
        role = 'UNKNOWN'
        code = 0
        my_state = {}
        try:
            role = open(RS_STATE).read().strip()
        except Exception:
            pass

        status = self.con.command('replSetGetStatus')
        for state in status['members']:
            if 'self' in state.keys():
                my_state = state
        # No changes.
        if my_state['stateStr'] == role:
            return Result(code=0, message='OK')
        # Master -> Secondary. Critical
        if role == 'PRIMARY' and my_state['stateStr'] != 'PRIMARY':
            with open(RS_STATE, 'w') as s:
                s.write(my_state['stateStr'])
            return Result(code=2, message='%s->%s' % (role, state['stateStr']))
        # Secondary -> anything else. A warning.
        if role != 'PRIMARY':
            with open(RS_STATE, 'w') as s:
                s.write(my_state['stateStr'])
            return Result(code=1, message='%s->%s' % (role, state['stateStr']))
    def slowlog(self, crit=None, warn=None):
        prev_slow_ops = 0

        if crit is None:
            crit = 100000
        if warn is None:
            warn = 100
        try:
            count = open(PROFILER_STATE).read().strip()
            if count.isdigit():
                prev_slow_ops = count
        except Exception:
            pass
        curr_slow_ops = self.con[self.db]['system.profile'].count()
        with open(PROFILER_STATE, 'w') as s:
            if str(curr_slow_ops).isdigit():
                s.write(str(curr_slow_ops))
        if curr_slow_ops >= crit:
            code = 2
        elif curr_slow_ops >= warn:
            code = 1
        else:
            code = 0
        return Result(code=code, message='%d slow ops at "%s"' % (int(curr_slow_ops), self.db))
    
if __name__ == '__main__':
    
    arg = argparse.ArgumentParser(description="""
            MongoDB checker.
            """
            )   
    arg.add_argument('-c', '--critical', type=int, required=False, metavar='<integer>',
            help='critical threshold')
    arg.add_argument('-w', '--warning', type=int, required=False, metavar='<integer>',
            help='warning threshold')
    arg.add_argument('-a', '--actions', required=True, nargs='+', metavar='<str>',
            help='perform these checks')
    arg.add_argument('-d', '--db', required=False, type=str, default='admin', metavar='<str>',
            help='use this database when applicable')
    arg.add_argument('-u', '--uri', required=True, type=str, metavar='<str>',
            help='URI to use when connecting')
    settings = vars(arg.parse_args())

    status = Status()
    try:
        check = CheckMongoDB(uri=settings.get('uri'), db=settings.get('db'))
        for action in settings.get('actions', []):
            try:
                func = getattr(check, action)
                result = func(crit=settings.get('critical'), warn=settings.get('warning'))
                status.append('%s: %s' % (action, result.message) )
                status.set_code(result.code)
            except Exception as e:
                # raise
                status.append('%s: %s' % (action, unicode(e)))
                status.set_code(2)
    except Exception as e:
        # raise
        status.report(code=2, message=unicode(e))
    
    status.report()
