#!/usr/bin/env python

import sys
import time

import psycopg2

STATE_FILE = '/tmp/pg_master.state'
TTL = 180


def die(code=0, comment="OK"):
    print('%d;%s' % (code, comment))
    sys.exit(0)


def get_postgres_master():
    try:
        conn = psycopg2.connect('dbname=postgres user=monitor connect_timeout=1')
        cur = conn.cursor()
        cur.execute("SELECT pg_is_in_recovery();")
        is_replica = cur.fetchone()
        if not is_replica or not is_replica[0]:
            return "{{ salt['grains.get']('id') }}"
        cur.execute("SELECT master FROM repl_mon;")
        master = cur.fetchone()[0]
        return master
    except psycopg2.OperationalError:
        die(1, "Could not connect to postgres")
    except Exception:
        die(1, "Could not get replication info")
    finally:
        try:
            cur.close()
            conn.close()
        except Exception:
            pass


def get_cached_master():
    try:
        info = open(STATE_FILE, 'r').read().split(';')
        return int(info[0]), info[1]
    except Exception:
        return []


def set_cached_master(master):
    ts = int(time.time())
    with open(STATE_FILE, 'w') as w:
        w.write('%d;%s' % (ts, master))
        return ts, master


if __name__ == '__main__':
    postgres_master = get_postgres_master()
    cached_info = get_cached_master() or set_cached_master(postgres_master)
    ts, cached_master = cached_info
    if cached_master != postgres_master:
        if time.time() - ts < TTL:
            die(code=2, comment='changed to %s' % postgres_master)
    set_cached_master(postgres_master)
    die(comment=postgres_master)
