#!/usr/bin/env python3
# -*- coding: utf-8 -%-
"""
Check if wal-e backup is not too old
"""

import argparse
import os
import sys
from datetime import datetime, timezone

from dateutil import parser

from wal_e import storage
from wal_e.blobstore import s3
from wal_e.exception import UserException
from wal_e.operator import s3_operator


def die(status=0, message='OK'):
    """
    Emit status and exit
    """
    print('%s;%s' % (status, message))
    sys.exit(0)


def _main():
    args_parser = argparse.ArgumentParser()
    args_parser.add_argument(
        '-w',
        '--warn',
        type=int,
        default=2,
        help='Warn if backup is older than this many days')
    args = args_parser.parse_args()
    try:
        creds = s3.Credentials(
            os.getenv('AWS_ACCESS_KEY_ID'),
            os.getenv('AWS_SECRET_ACCESS_KEY'),
            os.getenv('AWS_SECURITY_TOKEN'))
        store = storage.StorageLayout(os.getenv('WALE_S3_PREFIX'))
        s3_backup = s3_operator.S3Backup(store, creds,
                                         os.getenv('WALE_GPG_KEY_ID'))
        backup_list = list(s3_backup._backup_list(True))
        if not backup_list:
            die(1, 'No backups found')
        now = datetime.now(timezone.utc)
        latest_backup_delta = float('inf')
        for backup in backup_list:
            backup_time = parser.parse(backup.last_modified)
            delta = (now - backup_time).total_seconds()
            if delta < latest_backup_delta:
                latest_backup_delta = delta

        if latest_backup_delta < 86400 * args.warn:
            die()
        else:
            days = int(latest_backup_delta / 86400)
            die(1, 'Latest backup was more than %s days ago' % days)
    except UserException as exc:
        die(1, 'Unable to get status: %s, %s' % (exc.msg, exc.detail))
    except Exception as exc:
        die(1, 'Unable to get status: %s' % repr(exc))


if __name__ == '__main__':
    _main()
