#!/usr/bin/env python3
# -*- coding: utf-8 -%-
"""
Check if wal-g backup is not too old
"""

import argparse
import os
import sys
import subprocess
from datetime import datetime, timezone
from dateutil import parser

def die(status=0, message='OK'):
    """
    Emit status and exit
    """
    print('%s;%s' % (status, message))
    sys.exit(0)


def _main():
    args_parser = argparse.ArgumentParser()
    args_parser.add_argument(
        '-w',
        '--warn',
        type=int,
        default=2,
        help='Warn if backup is older than this many days')
    args = args_parser.parse_args()

    proc = subprocess.Popen(
                [
                    '/usr/bin/envdir', '/etc/wal-g/envdir',
                    '/usr/bin/timeout', '45',
                    '/usr/bin/wal-g', 'backup-list',
                ],stdout=subprocess.PIPE,stderr=subprocess.DEVNULL)

    latest_backup_delta = float('inf')
    now = datetime.now(timezone.utc)
    for line in iter(proc.stdout.readline, b''):
        tokens = line.rstrip().split()
        if len(tokens)<2:
            continue

        try:
            backup_time = parser.parse(tokens[1].decode("utf-8"))
        except ValueError:
            continue
        delta = (now - backup_time).total_seconds()
        if delta < latest_backup_delta:
            latest_backup_delta = delta

    if latest_backup_delta < 86400 * args.warn:
        die()
    elif latest_backup_delta == float('inf'):
        die(1,'No backups found')
    else:
        days = int(latest_backup_delta / 86400)
        die(1, 'Latest backup was more than %s days ago' % days)

if __name__ == '__main__':
    _main()
