#!/usr/bin/env python
"""
Check if local redis replication lag is not too large
"""

import argparse
import sys

from redis import StrictRedis

PASSWORD = '{{ salt['pillar.get']('data:redis:secrets:password') }}'
PORT = {{ salt['pillar.get']('data:redis:config:port', 6379) }}


def die(status=0, message='OK'):
    """
    Print status in monrun-compatible format
    """
    print('{status};{message}'.format(status=status, message=message))
    sys.exit(0)


def check_replication(conn, warn, crit):
    """
    Check replication lag against warn and crit thresholds
    """
    info = conn.info(section='replication')
    if info.get('role', 'slave') == 'slave':
        lag = info.get('master_last_io_seconds_ago', 0)
        if lag < warn:
            die()
        elif lag < crit:
            die(1, '{lag} seconds'.format(lag=lag))
        else:
            die(2, '{lag} seconds'.format(lag=lag))
    else:
        die()


def _main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-w', '--warn', type=int, default=10,
                        help='Warning threshold in seconds')
    parser.add_argument('-c', '--crit', type=int, default=30,
                        help='Critical threshold in seconds')
    args = parser.parse_args()
    for address in ['127.0.0.1', '::1']:
        try:
            conn = StrictRedis(
                host=address, port=PORT, db=0, password=PASSWORD)
            if conn.ping():
                return check_replication(conn, args.warn, args.crit)
        except Exception:
            pass
    die(1, 'Local redis is dead')


if __name__ == '__main__':
    _main()
