#!/usr/bin/env python3
# -*- coding: utf-8 -*-
import requests
from requests.adapters import HTTPAdapter
import json
import sys
import argparse


URL = 'https://%s/idm/ping/'
TIMEOUT = 1.0


def ping(url, cert_file, key_file, ca_bundle):
    ses = requests.Session()
    # http://docs.python-requests.org/en/master/api/#requests.adapters.HTTPAdapter
    ses.mount(
        'https://',
        HTTPAdapter(
            max_retries=1,
            pool_connections=1,
            pool_maxsize=1,
        )
    )
    try:
        # http://docs.python-requests.org/en/master/api/#requests.Session.request
        result = ses.get(
            url,
            cert=(cert_file, key_file),
            verify=ca_bundle,
            timeout=TIMEOUT,
        )
        code = result.status_code
        if code == 200:
            response = json.loads(result.text)
            if response['ping'] == 'pong':
                return code
            else:
                raise RuntimeError('bad response: %s' % response) 
        return code
    except requests.ConnectionError as exc:
        raise RuntimeError('unable to connect')
    except requests.Timeout:
        raise RuntimeError('read timed out (%s)' % TIMEOUT)
    except Exception:
        raise RuntimeError('unable to read response')


def result(code=0, msg='OK'):
    print(
        '{c};{m}'.format(
            c=code,
            m=msg
        )
    )
    sys.exit(0)


def parse_args():

    arg = argparse.ArgumentParser(description="""
        IDM-API pinger
        """
    )

    arg.add_argument(
        '-c', '--ca', metavar='<path>',
        required=True,
        help='path to CA bundle'
    )
    arg.add_argument(
        '-k', '--key', metavar='<path>',
        required=True,
        help='path to client certificate key'
    )
    arg.add_argument(
        '--host', metavar='<hostname>',
        required=True,
        help='hostname'
    )
    arg.add_argument(
        '--cert', metavar='<path>',
        required=True,
        help='path to client certificate'
    )
    return arg.parse_args()


def main():
    try:
        args = parse_args()
        actions = {
            200: [0, 'OK'],
            # 410 Gone -- https://tools.ietf.org/html/rfc2616#section-10.4.11
            410: [1, 'closed'],
        }
        code = ping(
            URL % args.host,
            cert_file=args.cert,
            key_file=args.key,
            ca_bundle=args.ca,
        )
        result(*actions.get(
            code, [2, 'err: got %s' % code]
        ))
    except Exception as exc:
        result(2, 'exc: %s' % exc)

        
if __name__ == '__main__':
    main()
