#! /usr/bin/env python3

import datetime
import msgpack
import os
import OpenSSL.crypto
import re


def juggler(results):
    resdict = {'CRIT': {},
               'WARN': {},
               'OK'  : {}}

    checkdesc = ''
    checkstatus = 0

    for key, value in results.items():
        if not value:
            # crit
            resdict['CRIT'][key] = 'Failed to parse'
        elif value < 5:
            resdict['CRIT'][key] = value
        elif value < 21:
            # warn
            resdict['WARN'][key] = value
        elif value < 30:
            # OK but show
            resdict['OK'][key] = value
        else:
            # OK, don't show
            pass

    if resdict['OK']:
        checkstatus = 0
        tmpdesc = 'OK: '
        for (handle, timing) in resdict['OK'].items():
            tmpdesc += '{}={} '.format(handle, timing)
        checkdesc = tmpdesc + checkdesc
    if resdict['WARN']:
        checkstatus = 1
        tmpdesc = 'WARN: '
        for (handle, timing) in resdict['WARN'].items():
            tmpdesc += '{}={} '.format(handle, timing)
        checkdesc = tmpdesc + checkdesc
    if resdict['CRIT']:
        checkstatus = 2
        tmpdesc = 'CRIT: '
        for (handle, timing) in resdict['CRIT'].items():
            tmpdesc += '{}={} '.format(handle, timing)
        checkdesc = tmpdesc + checkdesc
    print('{};{}'.format(checkstatus, checkdesc))


def get_cert_validity(certfile_data):
    crypto = OpenSSL.crypto
    cert_re = re.compile(r'(-----BEGIN CERTIFICATE-----.*?-----END CERTIFICATE-----)', re.DOTALL)
    m = cert_re.search(certfile_data)
    try:
        cert = crypto.load_certificate(crypto.FILETYPE_PEM, m.group(0))
    except:
        return None
    else:
        not_after = cert.get_notAfter()
        not_after = datetime.date(int(not_after[:4]), 
                                  int(not_after[4:6]),
                                  int(not_after[6:8]))
        today = datetime.date.today()
        valid_for = not_after - today
        return valid_for.days
    

def main():
    # cert validities dict
    validities = {}
    
    # populate dict from xconf dump
    xconf_dump_fname = '/var/xconf/mobile'
    f = open(xconf_dump_fname, 'rb')
    secrets = msgpack.unpackb(f.read())
    for name in secrets.keys():
        if name.startswith(b'apns:'):
            cert_data = secrets[name][0].decode('ascii', errors='ignore')
            validities[name[5:].decode("utf-8")] = get_cert_validity(cert_data)

    # populate dict from legacy files
    for dirpath, _, filenames in os.walk('/etc/yxiva-mobile/certs/apns/'):
        for fname in filenames:
            with open('/'.join((dirpath, fname)), 'r') as f:
                certfile_data = f.read()
                validities[fname] = get_cert_validity(cert_data)
    juggler(validities)
                
            


if __name__ == '__main__':
    main()

