#!/usr/bin/python

import subprocess
import sys
import urllib
import psycopg2

def subprocess_wrapper(cmd):
    p = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=sys.stderr)
    if p.wait() == 0:
        return p.communicate()[0].rstrip()
    else:
        print('Subprocess "%s" has exited with exit code %d' % (cmd, res))
        sys.exit(res)

def pg_query_wrapper(conn_string, query):
    try:
        conn = psycopg2.connect(conn_string)
        cur = conn.cursor()
        cur.execute(query)
        return cur.fetchone()[0]
    except Exception as err:
        print(err)
        return None

def get_list_of_clusters():
    cmd = "barman list-server"
    out = subprocess_wrapper(cmd)
    d = {}
    for line in out.splitlines():
        cluster, __tmp__, host = line.split()
        d[cluster] = host
    return d

def get_condgroup_of_host(host):
    condgroups = urllib.urlopen('http://c.yandex-team.ru/api/hosts2groups/%s' % host).read()
    condgroup = condgroups.splitlines()[-1]
    return condgroup

def get_all_hosts_of_cluster(cluster, condgroup):
    all_hosts_of_condgroup = urllib.urlopen('http://c.yandex-team.ru/api/groups2hosts/%s' % condgroup).read().splitlines()
    all_hosts_of_cluster = [host for host in all_hosts_of_condgroup if cluster in host]
    return all_hosts_of_cluster

def get_conninfo_for_cluster(cluster):
    cmd = "barman show-server %s | grep '[^_]conninfo' | cut -d' ' -f 2-" % cluster
    return subprocess_wrapper(cmd)

def pg_is_master(conn_string):
    query = "SELECT not (setting)::boolean FROM pg_settings WHERE name = 'transaction_read_only';"
    return pg_query_wrapper(conn_string, query)

def get_master_host(cluster, host):
    cgroup = get_condgroup_of_host(host)
    hosts = get_all_hosts_of_cluster(cluster, cgroup)
    conninfo = get_conninfo_for_cluster(cluster)
    if len(hosts) == 0:
        return conninfo
    for h in hosts:
        h_conn_string = conninfo.replace(host, h)
        if pg_is_master(h_conn_string):
            return h_conn_string

def get_actual_host(master_conn_string):
    query = "SELECT client_hostname FROM pg_stat_replication \
        WHERE state='streaming' ORDER BY sync_state, client_hostname DESC LIMIT 1;"
    try:
        return pg_query_wrapper(master_conn_string, query)
    except TypeError:
        return None

def change_config(config, old, new):
    cmd = "sed /etc/barman.yandex/%s -e 's/%s/%s/' >/etc/barman.d/%s" % \
            (config, old, new, config)
    __ = subprocess_wrapper(cmd)

if __name__ == '__main__':
    clusters = get_list_of_clusters()
    for cluster, host in clusters.items():
        master_conn_string = get_master_host(cluster, host)
        actual_host = get_actual_host(master_conn_string)
        if actual_host and actual_host != host:
            change_config('%s.conf' % cluster, host, actual_host)
