#!/usr/bin/env python
import json
import os
import socket
import time
import urllib2

PROJECT = 'disk'
ITYPES = {
    "disk": ":3132@disk",
    "rabbitmq": ":5672@rabbitmq",
    "zookeeper": ":2181@zookeeper",
}
INSTANCES = {
    "disk_all": ["disk"],
    "disk_backup": ["disk"],
    "disk_mqueue": ["rabbitmq"],
    "disk_zk": ["zookeeper"],
    "disk_zk_pgsync": ["zookeeper"],
}


class HTTPJSONCacher(object):
    def __init__(self, cache_file, ttl=300):
        self.cache_file = cache_file
        self.ttl = ttl

    def read(self):
        try:
            with open(self.cache_file, "r") as cache_fd:
                result = cache_fd.read()
        except Exception:
            result = None
        return result

    def update(self, data):
        try:
            cache_dir = os.path.dirname(self.cache_file)
            if not os.path.exists(cache_dir):
                os.mkdir(cache_dir)

            with open(self.cache_file, "w") as cache_fd:
                cache_fd.write(data)
        except Exception:
            return None

    def http_get(self, url, retries):
        for i in range(retries):
            try:
                http_req = urllib2.urlopen(url)
                if http_req.getcode() == 200:
                    return http_req.read()
            except Exception:
                pass
        return None

    def http_get_json(self, url, retries):
        json_data = dict()
        bdata = self.http_get(url, retries)
        json_data = json.loads(bdata)
        return json_data

    def get_json(self, url, retries=3):
        cache_needs_update = True
        if os.path.exists(self.cache_file):
            ts_now = time.time()
            ts_cache = os.path.getmtime(self.cache_file)
            if (ts_now - ts_cache) < self.ttl:
                cache_needs_update = False

        if cache_needs_update:
            json_data = self.http_get_json(url, retries)
            if json_data:
                self.update(json.dumps(json_data))

        cache_data = self.read()
        json_data = json.loads(cache_data)
        return json_data


class ConductorInfo(HTTPJSONCacher):
    def __init__(self):
        super(ConductorInfo, self).__init__('/tmp/conductor_cache_info.json')
        self.url = 'https://c.yandex-team.ru/api-cached/hosts/{0}/?format=json'

    def info(self, host):
        hostinfo = self.get_json(self.url.format(host))
        if not hostinfo:
            return
        return hostinfo[0]


class ConductorGroups(HTTPJSONCacher):
    def __init__(self):
        super(ConductorGroups, self).__init__('/tmp/conductor_cache_groups.json')
        self.url = 'https://c.yandex-team.ru/api-cached/hosts2groups/{0}/?format=json'

    def groups(self, host):
        groups = self.get_json(self.url.format(host))
        if not groups:
            return
        return groups


class Tags():
    def __init__(self):
        self.host = socket.getfqdn()
        info = ConductorInfo().info(self.host)
        self.ctype = info.get('group', '').replace('_', '-')
        self.datacenter = info.get('short_dc', info.get('root_datacenter', 'nodc'))
        self.itypes = {}
        for group in ConductorGroups().groups(self.host):
            group_name = group['name']
            if group_name not in INSTANCES:
                continue
            for itype in INSTANCES[group_name]:
                if itype not in ITYPES:
                    continue
                self.itypes[itype] = self.host + ITYPES[itype]

    def get(self):
        for itype, instance in self.itypes.items():
            res = [
                instance,
                'a_itype_' + itype,
                'a_prj_' + PROJECT,
                'a_ctype_' + self.ctype,
                'a_geo_' + self.datacenter,
            ]
            yield ' '.join(res)


if __name__ == '__main__':
    tags = Tags()
    for tag in tags.get():
        print(tag)
