CREATE SCHEMA disk;

CREATE EXTENSION IF NOT EXISTS "uuid-ossp";

CREATE TYPE disk.av_scan_status AS enum ('NEW', 'IN PROGRESS', 'CLEAN', 'INFECTED', 'ERROR');
CREATE TYPE disk.folder_type AS enum (
        'applications',
        'downloads',
        'google',
        'facebook',
        'fotki',
        'mailru',
        'photostream',
        'screenshots',
        'social',
        'instagram',
        'odnoklassniki',
        'vkontakte',
        'yabooks',
        'yalivelettersarchive',
        'yaruarchive',
        'yaslovariarchive'
);
CREATE TYPE disk.media_type AS enum (
        'audio',
        'backup',
        'compressed',
        'data',
        'development',
        'diskimage',
        'document',
        'encoded',
        'font',
        'image',
        'settings',
        'text',
        'video',
        'web',
        'executable',
        'spreadsheet',
        'flash',
        'book' ,
        'unknown'
);

CREATE TABLE disk.folders (
    uid                   bigint  NOT NULL,
    fid                   uuid    NOT NULL DEFAULT uuid_generate_v1(),
    parent_fid            uuid,
    id                    bytea,  -- NOT NULL,

    name                  text    NOT NULL,
    visible               boolean NOT NULL DEFAULT true,

    full_path             text,   -- For debug purposes only!

    version               bigint  NOT NULL DEFAULT 0,
    public                boolean NOT NULL DEFAULT FALSE,
    public_hash           bytea,
    symlink               text,
    short_url             text,
    blocked               boolean NOT NULL DEFAULT FALSE,
    published             boolean NOT NULL DEFAULT FALSE,

    files_count           integer NOT NULL DEFAULT 0,
    folders_count         integer NOT NULL DEFAULT 0,

    folder_type           disk.folder_type,

    date_removed          timestamp with time zone,  -- zdata.setprop.append_time
    date_uploaded         timestamp with time zone,  -- data.utime
    date_created          timestamp with time zone,  -- zdata.meta.ctime
    date_modified         timestamp with time zone,  -- data.mtime
    date_hidden_data      timestamp with time zone,  -- data.dtime

    path_before_remove    text,
    modify_uid            bigint,
    download_counter      bigint,
    folder_url            text,
    custom_properties     text,
    custom_setprop_fields jsonb,

    CONSTRAINT pk_folders PRIMARY KEY (uid, fid),
    CONSTRAINT fk_folders_uid_parent_fid FOREIGN KEY (uid, parent_fid)
        REFERENCES disk.folders ON DELETE NO ACTION DEFERRABLE INITIALLY DEFERRED,
    -- CONSTRAINT uk_folders_id UNIQUE (uid, id),
    CONSTRAINT check_uid CHECK (
        uid >= 0
    ),
    CONSTRAINT check_version CHECK (
        version >= 0
    ),
    CONSTRAINT check_parent_fid_not_zero CHECK (
        parent_fid != '00000000-0000-0000-0000-000000000000'::uuid
    ),
    CONSTRAINT check_empty_name CHECK (
        name != '' OR parent_fid IS NULL  -- Root has null parent_fid
    ),
    CONSTRAINT check_name_length CHECK (
        character_length(name) <= 1024
    )
    -- CONSTRAINT check_id_length CHECK (
    --     octet_length(id) = 32
    -- )

);

CREATE UNIQUE INDEX uk_folders_uid_parent_fid_name
    ON disk.folders
 USING btree (uid, parent_fid NULLS FIRST, name);  -- search begins from root, which always have parent_fid = NULL

CREATE UNIQUE INDEX uk_folders_uniq_root ON disk.folders
 USING btree (
    uid, fid, COALESCE(parent_fid, '00000000-0000-0000-0000-000000000000'::uuid)
 ) WHERE parent_fid IS NULL;


CREATE TABLE disk.files (
    uid                   bigint  NOT NULL,
    fid                   uuid    NOT NULL DEFAULT uuid_generate_v1(),
    parent_fid            uuid    NOT NULL,
    storage_id            uuid    NOT NULL,
    id                    bytea,  -- NOT NULL,

    full_path             text,   -- For debug purposes only!

    name                  text    NOT NULL,
    visible               boolean NOT NULL DEFAULT TRUE,
    version               bigint  NOT NULL DEFAULT 0,
    public                boolean NOT NULL DEFAULT FALSE,
    public_hash           bytea,
    symlink               text,
    short_url             text,
    download_counter      bigint,
    folder_url            text,
    blocked               boolean NOT NULL DEFAULT FALSE,
    source_uid            bigint,
    published             boolean NOT NULL DEFAULT FALSE,

    date_created          timestamp with time zone NOT NULL DEFAULT current_timestamp,  -- zdata.meta.ctime
    date_uploaded         timestamp with time zone NOT NULL DEFAULT current_timestamp,  -- data.utime
    date_modified         timestamp with time zone NOT NULL DEFAULT current_timestamp,  -- data.mtime
    date_removed          timestamp with time zone,  -- zdata.setprop.append_time
    date_hidden_data      timestamp with time zone,  -- data.dtime

    path_before_remove    text,
    modify_uid            bigint,

    media_type            disk.media_type,  -- data.mt
    mime_type             text,  -- data.mimetype

    source                text,  -- data.source
    custom_properties     text,
    custom_setprop_fields jsonb,

    CONSTRAINT pk_files PRIMARY KEY (uid, fid),
    CONSTRAINT uk_files_uid_parent_fid UNIQUE (uid, parent_fid, name),
    CONSTRAINT fk_files_uid_parent_fid FOREIGN KEY (uid, parent_fid)
        REFERENCES disk.folders (uid, fid) ON DELETE NO ACTION DEFERRABLE INITIALLY DEFERRED,
    -- CONSTRAINT fk_storage_files FOREIGN KEY (storage_id)
    --    REFERENCES disk.storage_files (storage_id) ON DELETE CASCADE DEFERRABLE INITIALLY DEFERRED,
    -- CONSTRAINT uk_files_id UNIQUE (uid, id),
    CONSTRAINT check_uid CHECK (
        uid >= 0
    ),
    CONSTRAINT check_version CHECK (
        version >= 0
    ),
    CONSTRAINT check_empty_name CHECK (
        name != ''
    ),
    CONSTRAINT check_name_length CHECK (
        character_length(name) <= 1024
    )
    -- CONSTRAINT check_id_length CHECK (
    --     octet_length(id) = 32
    -- )
);


CREATE TABLE disk.storage_files (
    storage_id     uuid   NOT NULL,

    stid           text   NOT NULL,
    digest_stid    text   NOT NULL,
    preview_stid   text,

    date_origin    timestamp with time zone NOT NULL DEFAULT current_timestamp,
    
    full_path      text,   -- For debug purposes only!

    size           bigint NOT NULL DEFAULT 0,

    md5_sum        uuid   NOT NULL,
    sha256_sum     bytea  NOT NULL,

    av_scan_status disk.av_scan_status NOT NULL DEFAULT 'NEW',
    date_exif      timestamp with time zone,
    video_data     jsonb,

    CONSTRAINT pk_storage_files PRIMARY KEY (storage_id),
    -- CONSTRAINT check_sha256_sum_length CHECK (
    --     octet_length(sha256_sum) = 32
    -- ),
    CONSTRAINT check_size CHECK (
        size >= 0
    )
);
