CREATE TABLE disk.filesystem_locks (
    id             uuid   NOT NULL,
    uid            bigint NOT NULL,
    path           text   NOT NULL,
    dtime          timestamp with time zone NOT NULL DEFAULT current_timestamp,
    data           jsonb,

    CONSTRAINT pk_filesystem_locks PRIMARY KEY (uid, path),
    CONSTRAINT uk_filesystem_locks_id UNIQUE (id),

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE TYPE disk.changelog_op_type AS ENUM (
    'new',
    'changed',
    'deleted'
);

CREATE TYPE disk.changelog_shared_type AS ENUM (
    'owner',
    'group',
    'unshared'
);

-- Definition is nowhere to be found in the original draft from disk.
-- Below is pure speculation.
CREATE TYPE disk.resource_type AS ENUM (
    'file',
    'dir'
);

CREATE TYPE disk.locales AS ENUM (
    'ru',
    'en',
    'tr',
    'uk'
);

CREATE TYPE disk.user_types AS ENUM (
    'standart',  -- its standarD, btw
    'attach'
);

CREATE TABLE disk.changelog (
    id          bytea                       NOT NULL,
    uid         bigint                      NOT NULL,
    path        text,
    type        disk.resource_type          NOT NULL,
    op          disk.changelog_op_type      NOT NULL,
    version     bigint                      NOT NULL,

    zdata       jsonb,

    gid         uuid,
    group_path  text,
    dtime       timestamp with time zone,

    shared      disk.changelog_shared_type,
    rights      int,

    CONSTRAINT pk_changelog PRIMARY KEY (id),

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE INDEX ix_changelog_dtime ON disk.changelog (dtime); -- Which direction?

CREATE TABLE disk.operations (
    id      bytea                    NOT NULL,
    uid     bigint                   NOT NULL,
    ctime   timestamp with time zone NOT NULL DEFAULT current_timestamp,
    dtime   timestamp with time zone NOT NULL DEFAULT current_timestamp,
    mtime   timestamp with time zone NOT NULL,
    state   int                      NOT NULL,
    version bigint                   NOT NULL,

    type    text,
    subtype text,
    md5     uuid,
    uniq_id uuid,

    data    jsonb,

    CONSTRAINT pk_operations PRIMARY KEY (id),
    
    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE INDEX ix_operations_uid ON disk.operations (uid);

CREATE TABLE disk.disk_info (
    id      uuid     NOT NULL,
    uid     bigint   NOT NULL,
    parent  uuid,
    version bigint,
    path    text     NOT NULL,
    type    disk.resource_type,
    data    jsonb,

    CONSTRAINT pk_disk_info PRIMARY KEY (uid, id),
    CONSTRAINT uk_disk_info_disk_info_id UNIQUE (id),

    CONSTRAINT fk_disk_info_id FOREIGN KEY (parent)
       REFERENCES disk.disk_info (id) ON DELETE CASCADE DEFERRABLE INITIALLY DEFERRED, -- Context reaction (DELETE, DO NOTHING)?

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE TABLE disk.user_index (
    uid         bigint NOT NULL,
    version     bigint NOT NULL,
    blocked     bool   NOT NULL DEFAULT FALSE,
    deleted     timestamp with time zone,
    
    user_type   disk.user_types DEFAULT 'standart',
    
    reg_time    timestamp with time zone,
    locale      disk.locales DEFAULT 'ru',
    
    shard_key   int NOT NULL, -- Still applicable with sharpei?
    b2b_key     text,
    pdd         jsonb,
    yateam_uid  bigint,

    collections text[] NOT NULL,

    CONSTRAINT pk_user_index PRIMARY KEY (uid),
    
    CONSTRAINT check_uid CHECK (
        uid >= 0
    ),
    CONSTRAINT check_version CHECK (
        version >= 0
    )
);

CREATE TABLE disk.groups (
    gid   uuid       NOT NULL,
    owner bigint     NOT NULL,
    path  text       NOT NULL,
    size  bigint     NOT NULL,

    CONSTRAINT pk_disk_groups PRIMARY KEY (gid),
    
    CONSTRAINT check_owner CHECK (
        owner >= 0
    )
);

CREATE TABLE disk.group_links (
    id      uuid     NOT NULL,
    gid     uuid     NOT NULL REFERENCES disk.groups (gid),
    path    text     NOT NULL,
    rights  int      NOT NULL,
    uid     bigint   NOT NULL,

    universe_login    text,
    universe_service  text,

    CONSTRAINT pk_group_links PRIMARY KEY (id),

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE TYPE disk.group_invites_status AS ENUM (
    'new',
    'activated',
    'rejected'
);

CREATE TABLE disk.group_invites (
    id         uuid   NOT NULL,
    avatar     text   NOT NULL DEFAULT '',
    gid        uuid   NOT NULL REFERENCES disk.groups (gid),
    name       text   NOT NULL DEFAULT '',
    rights     int    NOT NULL,
    uid        bigint NOT NULL,

    status     disk.group_invites_status NOT NULL DEFAULT 'new',
    ctime      timestamp with time zone NOT NULL DEFAULT current_timestamp,

    universe_login    text,
    universe_service  text,

    CONSTRAINT pk_group_invites PRIMARY KEY (id),

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE TABLE disk.link_data (
    id              uuid   NOT NULL,
    uid             bigint NOT NULL,
    path            text   NOT NULL,
    version         bigint,
    user_ip         inet,
    public_uid      bigint,

    target          text,

    type            disk.resource_type NOT NULL,
    parent          uuid,

    date_created    timestamp with time zone DEFAULT current_timestamp,
    date_modified   timestamp with time zone DEFAULT current_timestamp,
    date_deleted    timestamp with time zone,

    CONSTRAINT pk_link_data PRIMARY KEY (id),

    CONSTRAINT fk_link_data_parent FOREIGN KEY (parent)
       REFERENCES disk.link_data (id) ON DELETE CASCADE DEFERRABLE INITIALLY DEFERRED,

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE TYPE disk.album_item_type AS ENUM (
  'resource',
  'album',
  'shared_resource'
);

CREATE TYPE disk.album_layout AS ENUM (
  'squares',
  'rows',
  'waterfall',
  'fit_width'
);

CREATE TABLE disk.albums (
    id                bytea  NOT NULL,
    uid               bigint NOT NULL,
    title             text   NOT NULL,

    cover_id          bytea,
    cover_offset_y    real   DEFAULT 0.0,

    description       text,
    public_key        text,
    public_url        text,
    short_url         text,
    is_public         bool   NOT NULL DEFAULT FALSE,
    is_blocked        bool   NOT NULL DEFAULT FALSE,
    block_reason      text,
    flags             text[],
    layout            disk.album_layout,
    date_created      timestamp with time zone,
    date_modified     timestamp with time zone,
    social_cover_stid text,

    CONSTRAINT pk_albums PRIMARY KEY (uid, id),
    CONSTRAINT uk_albums_id UNIQUE (id),

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE TABLE disk.album_items (
    id          bytea                NOT NULL,
    uid         bigint               NOT NULL,
    album_id    bytea                NOT NULL,
    description text,
    group_id    uuid,
    order_index real                 DEFAULT 0.0,
    obj_id      text                 NOT NULL,
    obj_type    disk.album_item_type NOT NULL,

    CONSTRAINT pk_album_items PRIMARY KEY (uid, id),
    CONSTRAINT uk_album_items_id UNIQUE (id),

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

ALTER TABLE disk.albums ADD CONSTRAINT fk_albums_cover_id FOREIGN KEY (cover_id)
    REFERENCES disk.album_items (id) ON DELETE SET NULL DEFERRABLE INITIALLY DEFERRED;

ALTER TABLE disk.album_items ADD CONSTRAINT fk_album_items_album_id FOREIGN KEY (album_id)
    REFERENCES disk.albums (id) ON DELETE CASCADE DEFERRABLE INITIALLY DEFERRED;

CREATE TABLE disk.misc_data (
    id             uuid               NOT NULL,
    uid            bigint             NOT NULL,
    path           text               NOT NULL,
    type           disk.resource_type NOT NULL,

    parent         uuid,
    version        bigint,
    zdata          bytea,

    file_id        bytea,
    file_id_zipped bool,

    hid            uuid,
    mimetype       text,
    mediatype      disk.media_type,
    visible        bool,
    size           bigint,

    file_stid      text,
    digest_stid    text,
    preview_stid   text,

    date_modified  timestamp with time zone DEFAULT current_timestamp,
    date_uploaded  timestamp with time zone,
    date_exif      timestamp with time zone,

    CONSTRAINT pk_misc_data PRIMARY KEY (id),

    CONSTRAINT fk_misc_data_parent FOREIGN KEY (parent)
       REFERENCES disk.misc_data (id) ON DELETE CASCADE DEFERRABLE INITIALLY DEFERRED,

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);

CREATE TABLE disk.async_tasks_data (
    id           bytea                                              NOT NULL,
    uid          bigint                                             NOT NULL,
    date_created timestamp with time zone DEFAULT current_timestamp NOT NULL,
    data         bytea,

    CONSTRAINT pk_async_tasks_data PRIMARY KEY (id),

    CONSTRAINT check_uid CHECK (
        uid >= 0
    )
);
