CREATE OR REPLACE FUNCTION disk.process_live_photo_file_action() RETURNS TRIGGER AS $$
    BEGIN
        PERFORM * FROM disk.additional_file_links WHERE uid=NEW.uid AND main_file_fid=NEW.fid AND type='live_video';
        IF NOT FOUND THEN
            RAISE EXCEPTION 'Live photo link not found: uid %, fid %', NEW.uid, NEW.fid;
        END IF;
        RETURN NEW;
    END;
$$ LANGUAGE plpgsql VOLATILE;

CREATE CONSTRAINT TRIGGER file_update_live_photo
AFTER UPDATE ON disk.files
DEFERRABLE INITIALLY DEFERRED
FOR EACH ROW
WHEN (NEW.is_live_photo IS TRUE)
EXECUTE PROCEDURE disk.process_live_photo_file_action();

CREATE CONSTRAINT TRIGGER file_insert_live_photo
AFTER INSERT ON disk.files
DEFERRABLE INITIALLY DEFERRED
FOR EACH ROW
WHEN (NEW.is_live_photo IS TRUE)
EXECUTE PROCEDURE disk.process_live_photo_file_action();


CREATE OR REPLACE FUNCTION disk.process_live_photo_link_action() RETURNS TRIGGER AS $$
    BEGIN
        IF (TG_OP = 'DELETE') THEN
            PERFORM * FROM disk.files WHERE uid=OLD.uid AND fid=OLD.main_file_fid AND is_live_photo IS TRUE;
            IF FOUND THEN
                RAISE EXCEPTION 'File with is_live_photo flag set is found: uid %, fid %', OLD.uid, OLD.main_file_fid;
            END IF;
            RETURN OLD;
        ELSIF (TG_OP = 'UPDATE') THEN
            PERFORM * FROM disk.files WHERE uid=OLD.uid AND fid=OLD.main_file_fid AND is_live_photo IS TRUE;
            IF FOUND THEN
                RAISE EXCEPTION 'File with is_live_photo flag set is found: uid %, fid %', OLD.uid, OLD.main_file_fid;
            END IF;
            PERFORM * FROM disk.files WHERE uid=NEW.uid AND fid=NEW.main_file_fid AND is_live_photo IS NOT TRUE;
            IF FOUND THEN
                RAISE EXCEPTION 'File with is_live_photo flag is not set is found: uid %, fid %', NEW.uid, NEW.main_file_fid;
            END IF;
            RETURN NEW;
        END IF;
        RETURN NULL; -- result is ignored since this is an AFTER trigger
    END;
$$ LANGUAGE plpgsql VOLATILE;

CREATE CONSTRAINT TRIGGER link_delete_live_photo
AFTER DELETE ON disk.additional_file_links
DEFERRABLE INITIALLY DEFERRED
FOR EACH ROW
EXECUTE PROCEDURE disk.process_live_photo_link_action();

CREATE CONSTRAINT TRIGGER link_update_live_photo
AFTER UPDATE ON disk.additional_file_links
DEFERRABLE INITIALLY DEFERRED
FOR EACH ROW
WHEN (OLD.* IS DISTINCT FROM NEW.*)
EXECUTE PROCEDURE disk.process_live_photo_link_action();
